<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\File;

class VerifyTokenMiddleware
{
    public function handle(Request $request, Closure $next): Response
    {
        $allowedUris = [
            'api/validate-purchase',
            'api/verified-token',
            '/',
            'api/check-extensions',
            '/nlytical-install-1',
            '/nlytical-install-2',
            '/nlytical-install-3',
            '/nlytical-install-4',
        ];

        // Allow some URIs without token check
        if ($request->is($allowedUris)) {
            return $next($request);
        }

        // Verify token
        if ($this->verifyToken()) {
            return $next($request);
        }

        return response()->json([
            'success' => false,
            'message' => 'Unauthorized access. Invalid or missing token.'
        ], 400);
    }

    private function getMacAddress()
    {
        return exec('getmac');
    }

    private function getServerIP()
    {
        return request()->ip();
    }

    private function verifyToken()
    {
        try {
            $tokenFilePath = base_path('token/validatedToken.txt');

            if (!File::exists($tokenFilePath)) {
                Log::error("Token file not found: " . $tokenFilePath);
                return false;
            }

            $token = File::get($tokenFilePath);

            // No obfuscation, plain URL
            $apiUrl = "https://validator.whoxachat.com/verify_new";

            $response = Http::post($apiUrl, [
                "server_ip"   => $this->getServerIP(),
                "mac_address" => $this->getMacAddress(),
                "token"       => trim($token),
            ]);

            $data = $response->json();

            Log::info('Verification API Response:', $data);

            if (!isset($data['success']) || !$data['success']) {
                Log::error("Token verification failed");
                return false;
            }

            return true;
        } catch (\Exception $e) {
            Log::error('Verification Error:', ['exception' => $e->getMessage()]);
            return false;
        }
    }
}
