<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Controllers\BaseController;
use App\Models\Campaign;
use App\Models\Chat;
use App\Models\EmailVendorSwitch;
use App\Mail\VendorSwitched;
use App\Models\Service;
use App\Models\HomeSeo;
use App\Models\Category;
use Stripe\Stripe;
use Stripe\PaymentIntent;
use App\Models\GoalPayment;
use Illuminate\Support\Str;
use App\Models\Goals;
use Illuminate\Support\Facades\Mail;
use App\Models\NewsEmail;
use App\Models\PaymentGatewayKey;
use App\Models\PlanDescription;
use App\Models\ProfileBlockList;
use App\Models\ServiceCount;
use App\Models\ServiceImages;
use App\Models\ServiceLead;
use App\Models\ServiceLike;
use App\Models\SubCategory;
use App\Models\ServiceReview;
use App\Models\SiteSetup;
use App\Models\StoreAttachments;
use Illuminate\Support\Facades\Auth;
use Stripe\StripeClient;
use App\Models\StoreImages;
use App\Models\Stores;
use App\Models\UserSubscription;
use App\Models\SubscriptionPlan;
use App\Models\User;
use Carbon\Carbon;
use Twilio\Rest\Client;
use Exception;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Models\Lead;
use App\Models\GoalPaymentHistory;
use App\Models\UserSubscriptionHistory;

class VendorAuthController extends BaseController
{

    // vendorHome
    public function vendorHome(Request $request)
    {
        // $vendor_id = $request->vendor_id;
        $user = Auth::user();
        $vendor_id = $user->id;


        // Fetch vendor details
        $user = User::select('first_name', 'email')
            ->where('id', $vendor_id)
            ->first();

        if (!$user) {
            return response()->json([
                'status' => false,
                'message' => 'No vendor found',
                'first_name' => '',
                'serviceDetails' => '',
            ]);
        }

        $first_name = $user->first_name ?? '';
        $email = $user->email ?? '';

        $allServices = Service::select(
            'services.id',
            'services.vendor_id',
            'services.service_name',
            'services.service_description',
            'services.category_id',
            'services.subcategory_id',
            'services.address',
            'services.service_website',
            'services.service_country_code',
            'services.service_phone',
            'services.service_email',
            'services.lat',
            'services.lon',
            'services.open_days',
            'services.open_time',
            'services.close_time',
            'services.closed_days',
            'services.instagram_link',
            'services.facebook_link',
            'services.whatsapp_link',
            'services.twitter_link',
            'services.video_thumbnail',
            'services.video',
            'services.aspect_ratio',
            'services.request_approval',
            'services.employee_strength',
            'services.published_month',
            'services.published_year',
        )
            ->where('services.vendor_id', $vendor_id)
            ->orderBy('id', 'desc')
            ->get()
            ->transform(function ($service) {
                // Fetch service images along with their IDs
                $images = ServiceImages::where('service_id', $service->id)->get();
                $service->service_images = $images->isNotEmpty()
                    ? $images->map(function ($image) {
                        return [
                            'id' => $image->id, // Add image ID here
                            'url' => url('assets/images/service_images/' . $image->service_images)
                        ];
                    })->toArray()
                    : [['id' => 0, 'url' => url('assets/images/service_images/default_service.png')]]; // Default image ID can be null

                return $service;
            });

        // Check if there are services
        if ($allServices->isEmpty()) {
            return response()->json([
                'status' => true,
                'message' => 'Vendor Home List Found',
                'first_name' => $first_name,
                'serviceDetails' => [],
            ]);
        }

        // Preparing service details including contact details
        $serviceDetails = $allServices->map(function ($service) use ($email) {
            $images = ServiceImages::where('service_id', $service->id)->get();
            $service->service_images = $images->isNotEmpty()
                ? $images->map(function ($image) {
                    return [
                        'id' => $image->id,
                        'url' => url('assets/images/service_images/' . $image->service_images),
                    ];
                })->toArray()
                : [['id' => 0, 'url' => url('assets/images/service_images/default_service.png')]];


            // Fetch and format video thumbnail
            $service->video_thumbnail = $service->video_thumbnail
                ? url('assets/images/video_thumbnail/' . $service->video_thumbnail)
                : url('assets/images/video_thumbnail/default_service.png');

            // Fetch and format service video
            $service->video = $service->video
                ? url('assets/images/service_video/' . $service->video)
                : '';


            // Fetch total reviews and average rating
            $totalReviews = ServiceReview::where('service_id', $service->id)->get();
            $totalReviewStars = $totalReviews->sum('review_star');
            $totalReviewCount = $totalReviews->count();
            $avgReview = $totalReviewCount > 0 ? (string) round($totalReviewStars / $totalReviewCount, 1) : '';

            // Fetch category name
            $category = Category::find($service->category_id);
            $categoryName = $category ? $category->category_name : '';

            return [
                'businessDetails' => [  // Adding contact details here
                    'id' => $service->id,
                    'vendor_id' => $service->vendor_id,
                    'service_name' => $service->service_name ?? '',
                    'service_description' => $service->service_description ?? '',
                    'service_images' => $service->service_images,
                    'video_thumbnail' => $service->video_thumbnail,
                    'aspect_ratio' => $service->aspect_ratio ?? '',
                    'video' => $service->video,
                    'category_id' => $service->category_id ?? 0,
                    'subcategory_id' => $service->subcategory_id ?? '',
                    'request_approval' => $service->request_approval ?? 0,
                ],

                'contactDetails' => [  // Adding contact details here
                    'service_website' => $service->service_website ?? '',
                    'service_country_code' => $service->service_country_code ?? '',
                    'service_phone' => $service->service_phone ?? '',
                    'service_email' => $service->service_email ?? '', // Add email to contact details
                    'address' => $service->address ?? '',
                    'lat' => $service->lat ?? '',
                    'lon' => $service->lon ?? '',
                    'instagram_link' => $service->instagram_link ?? '',
                    'facebook_link' => $service->facebook_link ?? '',
                    'whatsapp_link' => $service->whatsapp_link ?? '',
                    'twitter_link' => $service->twitter_link ?? '',
                ],

                'businessTime' => [  // Adding contact details here
                    'open_days' => $service->open_days ?? '',
                    'open_time' => $service->open_time ?? '',
                    'close_time' => $service->close_time ?? '',
                    'closed_days' => $service->closed_days ?? '',
                    'employee_strength' => $service->employee_strength ?? '',
                    'published_month' => $service->published_month ?? '',
                    'published_year' => $service->published_year ?? '',
                ],

                'otherdetails' => [
                    'totalReviewCount' => (int) $totalReviewCount,
                    'totalAvgReview' => $avgReview,
                    'category_name' => $categoryName,
                ],
            ];
        })->toArray(); // Convert to array

        return response()->json([
            'status' => true,
            'message' => 'Vendor Home List Found',
            'first_name' => $first_name,
            'serviceDetails' => $serviceDetails,
        ]);
    }


    // businessReview
    public function businessReview(Request $request)
    {
        $service_id = $request->service_id;

        $service = Service::where(
            'id',
            $service_id
        )->first();
        if (!$service) {
            return response()->json([
                'status' => false,
                'message' => 'This Service Not Available',
            ]);
        }

        $serviceDetail = Service::select(
            'services.id',
            'services.service_name',
            'services.service_description',
        )
            ->where('services.id', $service_id)
            ->first();

        if (!$serviceDetail) {
            return response()->json([
                'status' => false,
                'message' => 'Service Not Found',
            ]);
        }

        // Fetch reviews for the service
        $productReviews = ServiceReview::select(
            'service_review.id',
            'service_review.service_id',
            'service_review.user_id',
            'service_review.review_star',
            'service_review.review_message',
            'service_review.created_at',
            'users.first_name',
            'users.last_name',
            'users.image'
        )
            ->leftJoin('users', 'service_review.user_id', '=', 'users.id')
            ->where('service_id', $service_id)
            ->orderBy('id', 'desc')
            ->get();

        // Fetch service images
        $serviceImages = ServiceImages::where('service_id', $service_id)->get();
        if ($serviceImages->isNotEmpty()) {
            $serviceDetail->service_images = $serviceImages->map(function ($image) {
                return url('assets/images/service_images/' . $image->service_images);
            });
        } else {
            $serviceDetail->service_images = [url('assets/images/service_images/default_service.png')];
        }

        // Calculate total reviews and average rating
        $totalReviewCount = $productReviews->count();
        $totalReviewStar = $productReviews->sum('review_star');
        $totalAvgReview = $totalReviewCount > 0 ? (string) round($totalReviewStar / $totalReviewCount, 1) : '';

        // Process reviews data
        $productReviews->transform(function ($review) {
            $review->first_name = $review->first_name ?? '';
            $review->last_name = $review->last_name ?? '';
            $review->image = $review->image
                ? url('assets/images/users_images/' . $review->image)
                : "";
            $review->created_at = $review->created_at ? $review->created_at : '';
            $review->time_ago = $review->created_at
                ? Carbon::parse($review->created_at)->diffForHumans()
                : '';
            return $review;
        });

        // Assign review data to service detail
        $serviceDetail->totalReviewCount = $totalReviewCount;
        $serviceDetail->totalAvgReview = $totalAvgReview;

        return response()->json([
            'status' => true,
            'message' => 'Service Detail Found',
            'service_details' => [
                'service_name' => $serviceDetail->service_name ?? '',
                'service_description' => $serviceDetail->service_description ?? '',
                'service_images' => $serviceDetail->service_images,
                'totalAvgReview' => $totalAvgReview,
                'totalReviewCount' => $totalReviewCount,
            ],
            'userReviews' => $productReviews,
        ]);
    }


    // addService
    public function addService(Request $request)
    {

        $user = Auth::user();
        $vendor_id = $user->id;

        if (!$vendor_id) {
            return response()->json(
                [
                    'status' => false,
                    'message' => 'Vendor ID is required',
                ],
                400
            );
        }

        $rules = [
            'service_name' => 'required',
            'service_description' => 'required',
            'category_id' => 'required|exists:categories,id',
            // 'subcategory_id' => 'required|exists:sub_categories,id',
            'service_phone' => 'required',
            'address' => 'required',
            'lat' => 'required',
            'lon' => 'required',
            'open_days' => 'required',
            // 'closed_days' => 'required',
            'service_images' => 'required',
            'open_time' => 'required',
            'close_time' => 'required',
            // 'area' => 'required',
            // 'city' => 'required',
            // 'state' => 'required',
            // 'country' => 'required',
            'cover_image' => 'required',
        ];

        $customMessages = [
            'service_name.required' => 'Please enter the service name.',
            'service_description.required' => 'Please enter the service description.',
            'category_id.required' => 'Please select a category.',
            'category_id.exists' => 'The selected category is invalid.',
            // 'subcategory_id.required' => 'Please select a subcategory.',
            // 'subcategory_id.exists' => 'The selected subcategory is invalid.',
            'service_phone.required' => 'Please enter the service phone number.',
            'address.required' => 'Please enter the address.',
            'lat.required' => 'Please enter the latitude.',
            'lon.required' => 'Please enter the longitude.',
            'open_days.required' => 'Please specify the open days.',
            // 'closed_days.required' => 'Please specify the closed days.',
            'service_images.required' => 'Please upload service images.',
            'open_time.required' => 'Please enter the opening time.',
            'close_time.required' => 'Please enter the closing time.',
            // 'area.required' => 'Please enter the area.',
            // 'city.required' => 'Please enter the city.',
            // 'state.required' => 'Please enter the state.',
            // 'country.required' => 'Please enter the country.',
            'cover_image.required' => 'Please select the cover image.',
        ];


        $validator = Validator::make($request->all(), $rules, $customMessages);
        if ($validator->fails()) {
            $errors = $validator->errors();

            $responseErrors = [];
            foreach ($errors->all() as $error) {
                $responseErrors = $error;
            }

            return response(
                [
                    'status' => false,
                    'message' => $responseErrors,
                ],
                401
            );
        }

        // Initialize $video_thumbnail and $video as null to handle cases where files are not provided
        $video_thumbnail = null;
        $video = null;
        $cover_image = null;

        if ($request->hasfile('cover_image')) {
            $originalName = pathinfo($request->cover_image->getClientOriginalName(), PATHINFO_FILENAME);
            $sanitizedFileName = Str::slug($originalName, '_') . '.' . $request->cover_image->getClientOriginalExtension();
            $cover_image = time() . '_' . $sanitizedFileName;
            $request->cover_image->move(public_path('/assets/images/cover_images/'), $cover_image);
        }

        if ($request->hasfile('video_thumbnail')) {
            $originalName = pathinfo($request->video_thumbnail->getClientOriginalName(), PATHINFO_FILENAME);
            $sanitizedFileName = Str::slug($originalName, '_') . '.' . $request->video_thumbnail->getClientOriginalExtension();
            $video_thumbnail = time() . '_' . $sanitizedFileName;
            $request->video_thumbnail->move(public_path('/assets/images/video_thumbnail/'), $video_thumbnail);
        }

        if ($request->hasfile('video')) {
            $originalName = pathinfo($request->video->getClientOriginalName(), PATHINFO_FILENAME);
            $sanitizedFileName = Str::slug($originalName, '_') . '.' . $request->video->getClientOriginalExtension();
            $video = time() . '_' . $sanitizedFileName;
            $request->video->move(public_path('/assets/images/service_video/'), $video);
        }


        // Update the role of the user associated with the vendor_id to 'vendor'
        $user = User::where('id', $vendor_id)->first();
        if ($user) {
            $user->role = 'vendor';
            $user->save();
        }


        $data = new Service();
        $data->vendor_id = $vendor_id;
        $data->category_id = $request->category_id;
        $data->subcategory_id = $request->subcategory_id;
        $data->service_name = $request->service_name;
        $data->service_description = $request->service_description;
        $data->service_website = $request->service_website;
        $data->service_country_code = $request->service_country_code;
        $data->service_phone = $request->service_phone;
        $data->lat = $request->lat;
        $data->lon = $request->lon;
        $data->service_address = $request->service_address;
        $data->address = $request->address;
        $data->open_days = $request->open_days;
        $data->closed_days = $request->closed_days;
        $data->open_time = $request->open_time;
        $data->close_time = $request->close_time;
        $data->instagram_link = $request->instagram_link;
        $data->facebook_link = $request->facebook_link;
        $data->whatsapp_link = $request->whatsapp_link;
        $data->twitter_link = $request->twitter_link;
        $data->video_thumbnail = $video_thumbnail;
        $data->video = $video;
        $data->aspect_ratio = $request->aspect_ratio;
        $data->employee_strength = $request->employee_strength;
        $data->published_month = $request->published_month;
        $data->published_year = $request->published_year;
        $data->is_featured = $request->is_featured;
        $data->service_email = $request->service_email;
        $data->area = $request->area;
        $data->city = $request->city;
        $data->state = $request->state;
        $data->country = $request->country;
        $data->meta_title = $request->meta_title;
        $data->meta_description = $request->meta_description;
        $data->video_url = $request->video_url;
        $data->cover_image = $cover_image;
        $data->save();

        if ($request->hasfile('service_images')) {
            foreach ($request->file('service_images') as $file) {
                $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();

                $fileName = time() . '_' . $sanitizedFileName;
                $file->move(public_path('/assets/images/service_images/'), $fileName);

                $data->serviceImages()->create([
                    'service_id' => $data->id,
                    'service_images' => $fileName,
                ]);
            }
        }

        // **Send Vendor Switch Email**
        // $emailPreference = EmailVendorSwitch::where('get_email', 1)->first();
        // if ($emailPreference && $user && !empty($user->email)) {
        //     Mail::to($user->email)->send(new VendorSwitched($user->email, $user->first_name));
        // }


        return response(
            [
                'status' => true,
                'message' => 'Service added successfully...!',
                'service_id' => $data->id,
            ],
            200
        );
    }


    // updateService
    public function updateService(Request $request)
    {
        try {
            // $vendor_id = $request->vendor_id;
            $user = Auth::user();
            $vendor_id = $user->id;
            $service_id = $request->service_id;

            // Check if vendor_id is provided
            if (!$vendor_id) {
                return response([
                    'status' => false,
                    'message' => 'Vendor ID is required.',
                ], 404);
            }

            // Find the service
            $service = Service::find($service_id);

            // Check for existence and authorization
            if (!$service || $service->vendor_id != $vendor_id) {
                return response([
                    'status' => false,
                    'message' => 'Service not found or not authorized.',
                ], 404);
            }


            $updateData = $request->only([
                'service_name',
                'service_description',
                'category_id',
                'subcategory_id',
                'service_website',
                'service_country_code',
                'service_phone',
                'address',
                'lat',
                'lon',
                'open_days',
                'service_address',
                'closed_days',
                'open_time',
                'close_time',
                'instagram_link',
                'facebook_link',
                'twitter_link',
                'whatsapp_link',
                'aspect_ratio',
                'employee_strength',
                'published_month',
                'published_year',
                'service_email',
                'area',
                'city',
                'state',
                'country',
                'cover_image',
                'meta_title',
                'meta_description',
                'video_url',
            ]);

            // If closed_days is an empty string or not provided, set it to null
            if (empty($request->closed_days)) {
                $updateData['closed_days'] = null;
            }

            // Update the service
            $service->update($updateData);

            // Handle service images
            if ($request->hasFile('service_images')) {
                foreach ($request->file('service_images') as $file) {
                    $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                    $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                    $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                    $file->move(public_path('assets/images/service_images'), $fileName);

                    $service->serviceImages()->create([
                        'service_images' => $fileName,
                    ]);
                }
            }

            // Handle video thumbnail upload and update column
            if ($request->hasFile('video_thumbnail')) {
                $file = $request->file('video_thumbnail');
                $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                $file->move(public_path('assets/images/video_thumbnail'), $fileName);

                $service->update([
                    'video_thumbnail' => $fileName,
                ]);
            }

            // Handle video upload and update column
            if ($request->hasFile('video')) {
                $file = $request->file('video');
                $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                $file->move(public_path('assets/images/service_video'), $fileName);

                $service->update([
                    'video' => $fileName,
                ]);
            }

            // Handle cover image upload and update column
            if ($request->hasFile('cover_image')) {
                $file = $request->file('cover_image');
                $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                $file->move(public_path('assets/images/cover_images'), $fileName);

                $service->update([
                    'cover_image' => $fileName,
                ]);
            }

            // Get service images and video (with full paths)
            // Get service images with ID and full paths
            $serviceImages = $service->serviceImages()->get(['id', 'service_images'])->map(function ($image) {
                return [
                    'id' => $image->id,
                    'service_images' => asset('assets/images/service_images/' . $image->service_images), // Full path
                ];
            });


            $serviceVideo = $service->video ? asset('assets/images/service_video/' . $service->video) : '';

            $serviceCoverimage = $service->cover_image ? asset('assets/images/cover_images/' . $service->cover_image) : '';

            $serviceVideoThumb = $service->video_thumbnail ? asset('assets/images/video_thumbnail/' . $service->video_thumbnail) : '';


            // Replace null values with empty strings
            $service = $service->toArray();
            array_walk_recursive($service, function (&$value) {
                if (is_null($value)) {
                    $value = '';
                }
            });

            // Return response with updated data, including service_images and service_video
            return response([
                'status' => true,
                'message' => 'Service updated successfully!',
                'service' => $service,  // Include the updated service data with empty strings instead of null
                'service_images' => $serviceImages,  // Include service images with full paths
                'video' => $serviceVideo,  // Include service video with full path
                'cover_image' => $serviceCoverimage,  // Include service video with full path
                'video_thumbnail' => $serviceVideoThumb,  // Include service video with full path


            ], 200);
        } catch (\Exception $e) {
            return response([
                'status' => false,
                'message' => 'An error occurred while updating the service: ' . $e->getMessage(),
            ], 500);
        }
    }


    // addChat
    public function addChat(Request $request)
    {
        $user = Auth::user();
        $from_user = $user->id;
        $rules = [
            // 'from_user' => 'required|exists:users,id',
            'to_user' => 'required|exists:users,id',
            'message' => 'nullable|string',
            'url' => 'nullable|file',
            'type' => 'nullable',
        ];

        $customMessages = [
            // 'from_user.required' => 'Please enter from user id.',
            'to_user.required' => 'Please enter to user id.',
        ];

        $validator = Validator::make($request->all(), $rules, $customMessages);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'message' => $validator->errors()->first(),
            ], 422);
        }

        // $from_user = $request->from_user;
        $to_user = $request->to_user;

        // Check if the users are blocked
        $isBlocked = ProfileBlockList::where('blockedByUserId', $from_user)
            ->where('blockedUserId', $to_user)
            ->exists();

        if ($isBlocked) {
            return response()->json([
                'status' => false,
                'message' => 'You have blocked this user',
            ], 403);
        }

        $fileName = null;
        $fileType = 'text';

        // Check if a file is uploaded
        if ($request->hasFile('url')) {
            $file = $request->file('url');
            $mimeType = $file->getMimeType();

            // Check if the file is an image or document
            if (strpos($mimeType, 'image') !== false) {
                $fileName = time() . '_' . $file->getClientOriginalName();
                $file->move(public_path('assets/images/chat_images'), $fileName);
                $fileType = 'image';
            } elseif (strpos($mimeType, 'pdf') !== false || strpos($mimeType, 'doc') !== false) {
                $fileName = time() . '_' . $file->getClientOriginalName();
                $file->move(public_path('assets/images/chat_images'), $fileName);
                $fileType = 'doc';
            } else {
                return response()->json([
                    'status' => false,
                    'message' => 'Unsupported file type',
                ], 422);
            }
        }

        $type = $request->type;
        $is_lead = $request->is_lead;

        $chat = new Chat();
        $chat->from_user = $from_user;
        $chat->to_user = $to_user;
        $chat->message = $request->message ?? '';
        $chat->url = $fileName ?? '';
        $chat->type = $fileType;
        $chat->is_lead = $is_lead ?? '';
        $chat->save();

        if ($is_lead) {

            Lead::create([
                'from_user' => $from_user,
                'to_user' => $to_user,
            ]);
        }

        $FcmToken = User::where('id', $to_user)->value('device_token');
        $username = User::where('id', $from_user)->first();
        $first_name = $username->first_name;
        $last_name = $username->last_name;
        $last_seen = $username->updated_at;
        $block_status = $isBlocked ? 1 : 0;


        $profile_image = $username->image ? url('assets/images/users_images/' . $username->image)
            : "";

        // $profile_image = url('public/images/user/' . $username->profile_pic);

        $data = [
            'title' => "Message",
            'message' => "$first_name sent you a message.",
            'type' => "Message",
            'from_user' => $from_user,
            'to_user' => $to_user,
            'profile_image' => $profile_image,
            'first_name' => $first_name,
            'last_name' => $last_name,
            'last_seen' => $username->updated_at->toDateTimeString(),
            'block_status' => $block_status,

        ];

        // dd($data);

        // $this->sendchatNotification($data, $FcmToken);
        $baseController = new BaseController();
        $baseController->sendchatNotification(new Request($data), $FcmToken);


        return response()->json([
            'status' => true,
            'message' => 'Chat message sent successfully',
        ]);
    }


    // innerChat
    public function innerChat(Request $request)
    {
        $user = Auth::user();
        $from_user = $user->id;
        // Validation rules
        $rules = [
            // 'from_user' => 'required|exists:users,id',
            'to_user' => 'required|exists:users,id',
        ];

        // Custom validation messages
        $customMessages = [
            // 'from_user.required' => 'Please enter from user ID.',
            'to_user.required' => 'Please enter to user ID.',
        ];

        // Validate the request
        $validator = Validator::make($request->all(), $rules, $customMessages);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'errors' => $validator->errors()], 400);
        }

        // $from_user = $request->from_user;
        $to_user = $request->to_user;

        // Get user details (from_user)
        $user = User::select('id', 'first_name', 'last_name', 'image', 'role', 'updated_at') // Added 'role'
            ->where(
                'id',
                $from_user
            )
            ->get()
            ->transform(function ($user) {
                $user['image'] = $user['image'] ? url('assets/images/users_images/' . $user['image']) : "";
                return $user;
            })
            ->toArray();

        // Get user details (to_user)
        $toUserDetails = User::select('id', 'first_name', 'last_name', 'image', 'role', 'updated_at', 'status', 'updated_at') // Added 'role'
            ->where(
                'id',
                $to_user
            )
            ->first();

        // Handle to_user image URL
        $toUserImage = $toUserDetails->image ? url('assets/images/users_images/' . $toUserDetails->image)
            : "";

        // Update message read status for chats where to_user is from_user
        Chat::where('to_user', $from_user)
            ->where('from_user', $to_user)
            ->update(['read_message' => '1']);

        // Fetch chat messages between from_user and to_user
        $data = Chat::where(function ($query) use ($from_user, $to_user) {
            $query->where('from_user', $from_user)
                ->where('to_user', $to_user)
                ->orWhere(function ($query) use ($from_user, $to_user) {
                    $query->where('from_user', $to_user)
                        ->where('to_user', $from_user);
                });
        })
            ->orderBy('created_at', 'desc')
            ->get()
            ->groupBy(function ($message) {
                return $message->created_at->format('Y-m-d');
            })
            ->map(function ($groupedMessages, $date) {
                // Format the date in ISO 8601 format
                $formattedDate = Carbon::createFromFormat('Y-m-d', $date)->format('Y-m-d\TH:i:s.u\Z');
                return [
                    'date' => $formattedDate,
                    'messages' => $groupedMessages->map(function ($message) {
                        // Fetch the profile of the sender (from_user)
                        $from_user_profile = User::select('first_name', 'last_name', 'image', 'role', 'status') // Added 'role'
                            ->where('id', $message->from_user)
                            ->first();

                        // Handle profile image URL
                        $profile_image = $from_user_profile->image ? url('assets/images/users_images/' . $from_user_profile->image) : "";

                        // Format the chat time
                        $chat_time = Carbon::createFromFormat('Y-m-d H:i:s', $message->created_at)->format('h:i A');

                        $is_lead = Lead::where('from_user', $message->from_user)->where('to_user', $message->to_user)->first();
                        $is_lead_done = $is_lead ? "1" : "0";

                        // Return the formatted message details
                        return [
                            'id' => (int)$message->id,
                            'from_user' => (int)$message->from_user,
                            'to_user' => (int)$message->to_user,
                            'message' => $message->message ?? "",
                            'url' => $message->url ? url('assets/images/chat_images/' . $message->url) : "",
                            'type' => $message->type ?? "",
                            'read_message' => (string)($message->read_message ?? ""),
                            'created_at' => $message->created_at ?? "",
                            'first_name' => $from_user_profile->first_name ?? "",
                            'last_name' => $from_user_profile->last_name ?? "",
                            'role' => $from_user_profile->role ?? "", // Include the role
                            'is_online' => $from_user_profile->status ?? 0, // Include the role
                            'profile_image' => $profile_image,
                            'message_seen' => (int)($message->read_message ?? ""),
                            'chat_time' => $chat_time,
                            'is_lead' => $is_lead_done,
                        ];
                    }),
                ];
            });

        // Prepare response
        $response = [
            'success' => true,
            'message' => "Message list.",
            'to_user_details' => [
                'id' => (int)$toUserDetails->id,  // Add the ID here
                'first_name' => $toUserDetails->first_name ?? "",
                'last_name' => $toUserDetails->last_name ?? "",
                'is_online' => $toUserDetails->status,
                'image' => $toUserImage,
                'role' => $toUserDetails->role ?? "", // Include the role
                'updated_at' => $toUserDetails->updated_at->diffForHumans() ?? "",
            ],
            'chatMessages' => $data->values()->all(),
        ];

        return response()->json($response, 200);
    }


    // userChatList
    public function userChatList(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        // $user_id = $request->user_id;
        $search = $request->first_name; // Get the first_name from the request

        // Query to get the users that match with the user's chat history
        if ($search) {
            $data = User::select('id', 'first_name', 'last_name', 'image', 'status', 'role')
                ->where(function ($query) use ($user_id) {
                    $query->whereIn('id', function ($q) use ($user_id) {
                        $q->select('to_user')->from('chats')->where('from_user', $user_id);
                    })->orWhereIn('id', function ($q) use ($user_id) {
                        $q->select('from_user')->from('chats')->where('to_user', $user_id);
                    });
                });

            // If the search term is provided, filter users by first_name starting with the first 3 characters
            $searchQuery = substr($search, 0, 3); // Get the first 3 characters
            $data->where('first_name', 'LIKE', $searchQuery . '%');
        } else {
            $data = User::select('id')->whereIn('id', function ($q) use ($user_id) {
                $q->select('to_user')->from('chats')->where('from_user', $user_id);
            })->orwhereIn('id', function ($q) use ($user_id) {
                $q->select('from_user')->from('chats')->where('to_user', $user_id);
            });
        }

        // Retrieve the data
        $data = $data->get();

        $data_Ar = array();
        foreach ($data as $row) {
            $last_message_query = Chat::where(function ($q) use ($user_id, $row) {
                $q->where('from_user', $user_id)
                    ->where('to_user', $row->id);
            })->orWhere(function ($q) use ($user_id, $row) {
                $q->where('from_user', $row->id)
                    ->where('to_user', $user_id);
            })->orderBy('created_at', 'DESC')->first();

            $chat_list['id'] = (int)$last_message_query->id;
            $chat_list['my_id'] = (int)$user_id;
            $chat_list['second_id'] = (int)$row->id;

            // Determine the last message type and URL
            $chat_list['last_message'] = $last_message_query->type == 'image' ? 'image' : ($last_message_query->type == 'document' ? 'document' : $last_message_query->message ?? "");
            $chat_list['url'] = $last_message_query->url ? url('images/chat/' . $last_message_query->url) : "";
            $chat_list['type'] = $last_message_query->type ?? "";

            $user = $last_message_query->to_user == $user_id ? User::where('id', $last_message_query->from_user)->first() : User::where('id', $last_message_query->to_user)->first();
            $chat_list['user_id'] = (int)$user->id;
            $chat_list['first_name'] = $user->first_name ?? "";
            $chat_list['last_name'] = $user->last_name ?? "";
            $chat_list['is_online'] = $user->status ?? 0;
            $chat_list['user_role'] = $user->role ?? "";
            $chat_list['profile_pic'] = $user->image ? url('assets/images/users_images/' . $user->image) : "";
            $chat_list['time'] = $last_message_query->created_at->diffForHumans();
            $chat_list['last_seen'] = $user->updated_at->diffForHumans();

            $is_lead = Lead::where('from_user', $user_id)->where('to_user', $row->id)->first();
            $is_lead_done = $is_lead ? "1" : "0";

            $chat_list['is_lead'] = $is_lead_done;

            // Check if the user is blocked
            $is_blocked = ProfileBlockList::where(function ($query) use ($user_id, $row) {
                $query->where('blockedByUserId', $user_id)
                    ->where('blockedUserId', $row->id);
            })->orWhere(function ($query) use ($user_id, $row) {
                $query->where('blockedByUserId', $row->id)
                    ->where('blockedUserId', $user_id);
            })->exists();

            $chat_list['is_block'] = $is_blocked ? 1 : 0;

            $chat_list['unread_message'] = (string)Chat::where('to_user', $user_id)->where('from_user', $row->id)->where('read_message', 0)->count();

            array_push($data_Ar, $chat_list);
        }

        // Sorting chat list by the most recent chat (using message ID)
        $chat = array();
        foreach ($data_Ar as $key => $row) {
            $chat[$key] = $row['id'];
        }
        array_multisort($chat, SORT_DESC, $data_Ar);

        $response = [
            'success' => true,
            'message' => "Chat Message list.",
            'chat_list' => $data_Ar
        ];

        return response()->json($response, 200);
    }


    // subscriptionPlan
    public function subscriptionPlanold(Request $request)
    {
        // Fetch all subscription plans
        $all_plan_list = SubscriptionPlan::get();

        // Fetch default currency and currency value
        $siteSetup = SiteSetup::first();
        $defaultCurrency = $siteSetup ? $siteSetup->default_currency : '';
        $currencyValue = $siteSetup ? $siteSetup->currency_value : '';


        $vertical = array();

        foreach ($all_plan_list as $row) {
            $res = [];
            $res['id'] = (int)$row->id;
            $res['plan_name'] = $row->plan_name ? $row->plan_name : "";
            $res['description'] = $row->description ? $row->description : "";
            $res['price'] = ($row->price ? $defaultCurrency . $row->price : ""); // Append currency
            $res['duration'] = $row->duration ? $row->duration : "";
            $res['currency_value'] = $currencyValue; // Corrected variable name
            $res['subtext'] = $row->subtext ? $row->subtext : "";


            // Push the plan with its description to the result array
            array_push($vertical, $res);
        }

        return response()->json([
            'status' => true,
            'message' => 'Subscription Plan Found',
            'subscriptionDetail' => $vertical,
        ]);
    }

    public function subscriptionPlan(Request $request)
    {
        $search = $request->input('search');
        $page = (int) $request->input('page', 1);
        $pageSize = (int) $request->input('pageSize', 10);
        $orderColumn = $request->input('orderColumn', 'id');
        $orderDir = strtoupper($request->input('orderDir', 'DESC'));

        // Whitelist allowed order columns
        $allowedOrderColumns = ['id', 'plan_name', 'price', 'validity_day', 'created_at'];
        if (!in_array($orderColumn, $allowedOrderColumns)) {
            $orderColumn = 'id';
        }
        if (!in_array($orderDir, ['ASC', 'DESC'])) {
            $orderDir = 'DESC';
        }

        $query = SubscriptionPlan::query();

        if (!empty($search)) {
            $query->where('plan_name', 'like', '%' . $search . '%');
        }

        $total = $query->count();

        $plans = $query->orderBy($orderColumn, $orderDir)
            ->skip(($page - 1) * $pageSize)
            ->take($pageSize)
            ->get();

        $siteSetup = SiteSetup::first();
        $defaultCurrency = $siteSetup?->default_currency ?? '';
        $currencyValue = $siteSetup?->currency_value ?? '';
        $currencyDone = $siteSetup?->default_currency_name ?? '';

        $result = [];

        foreach ($plans as $row) {
            $result[] = [
                'id' => (int)$row->id,
                'plan_name' => $row->plan_name ?? '',
                'description' => $row->description ?? '',
                'price' => $row->price ? $defaultCurrency . $row->price : '',
                'duration' => $row->duration ?? '',
                'currency_value' => $currencyValue,
                'subtext' => $row->subtext ?? '',
                'created_at' => $row->created_at ?? '',
                'currency_name' => $currencyDone,
            ];
        }

        return response()->json([
            'status' => !empty($result),
            'message' => !empty($result) ? 'Subscription Plan Found' : 'No Plans Found',
            'subscriptionDetail' => $result,
            'Pagination' => [
                'total_pages' => ceil($total / $pageSize),
                'total_records' => $total,
                'current_page' => $page,
                'records_per_page' => $pageSize,
            ],
        ]);
    }


    // public function subscriptionPlan(Request $request)
    // {
    //     \Stripe\Stripe::setApiKey('sk_test_51OP303SJayPbST1lMDr6nn6WieehdLmIpiG2pgVil38DVNPjDqKcFG87d1GMOk10WWtoqZIxvSx2WLAP7G1GMkWu00SJWzq7cn');
    //     $stripe = new \Stripe\StripeClient('sk_test_51OP303SJayPbST1lMDr6nn6WieehdLmIpiG2pgVil38DVNPjDqKcFG87d1GMOk10WWtoqZIxvSx2WLAP7G1GMkWu00SJWzq7cn');

    //     // Fetch local plans
    //     $localPlans = SubscriptionPlan::get();

    //     // Fetch Stripe products and prices
    //     $stripeProducts = $stripe->products->all(['active' => true]);
    //     $mergedPlans = [];

    //     // Fetch default currency and currency value from local DB
    //     $siteSetup = SiteSetup::first();
    //     $defaultCurrency = $siteSetup ? $siteSetup->default_currency : '';
    //     // $currencyValue = $siteSetup ? $siteSetup->currency_value : '';

    //     foreach ($localPlans as $local) {
    //         $matchedProduct = collect($stripeProducts->data)->first(function ($prod) use ($local) {
    //             return strtolower(trim($prod->name)) == strtolower(trim($local->plan_name));
    //         });

    //         $priceData = null;

    //         if ($matchedProduct) {
    //             $prices = $stripe->prices->all([
    //                 'product' => $matchedProduct->id,
    //                 'active' => true,
    //                 'limit' => 1,
    //             ]);

    //             if (count($prices->data) > 0) {
    //                 $priceData = $prices->data[0];
    //             }
    //         }

    //         $mergedPlans[] = [
    //             'id' => (int) $local->id,
    //             'plan_name' => $local->plan_name ?? '',
    //             'description' => $local->description ?? '',
    //             'subtext' => $local->subtext ?? '',
    //             'validity_day' => $local->validity_day ?? '',
    //             // 'currency_value' => $currencyValue,
    //             'price' => $local->price ? $defaultCurrency . $local->price : '',
    //             'stripe_product_id' => $matchedProduct->id ?? null,
    //             'price_id' => $priceData->id ?? null,
    //             'stripe_currency' => isset($priceData->currency) ? strtoupper($priceData->currency) : null,
    //             'amount' => isset($priceData->unit_amount) ? number_format($priceData->unit_amount / 100, 2) : null,
    //             'interval' => $priceData->recurring->interval ?? null,
    //         ];
    //     }

    //     return response()->json([
    //         'status' => true,
    //         'message' => 'Merged Subscription Plans with Stripe',
    //         'subscriptionDetail' => $mergedPlans,
    //     ]);
    // }





    // paymentSuccess
    public function paymentSuccess(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $plan_name = $request->input('plan_name');
        $payment_mode = $request->input('payment_mode');
        $price = $request->input('price');
        $subscription_id = $request->input('subscription_id');


        if (
            !$user_id || !$plan_name || !$payment_mode || !$price || !$subscription_id
        ) {
            return response()->json([
                'status' => false,
                'message' => 'Invalid input data',
            ], 400);
        }

        $check_plan = SubscriptionPlan::where('id', $subscription_id)->first();
        if (!$check_plan) {
            return response()->json([
                'status' => false,
                'message' => 'Subscription id not found',
            ], 404);
        }

        // if (
        //     $plan_name == "Yearly"
        // ) {
        //     $expire_date = now()->addYear()->format('Y-m-d');
        // } else {
        //     $expire_date = now()->addMonth()->format('Y-m-d');
        // }

        $start_date = now();

        $duration = $check_plan->duration; // e.g. "1 Month", "3 Month", "1 Year"
        $expire_date = $start_date;

        if ($duration) {
            // Extract numeric value and unit from duration string
            preg_match('/(\d+)\s*(Month|Year)/i', $duration, $matches);

            if (isset($matches[1]) && isset($matches[2])) {
                $value = (int)$matches[1];
                $unit = strtolower($matches[2]);

                if ($unit == 'month') {
                    $expire_date = $start_date->copy()->addMonths($value);
                } elseif ($unit == 'year') {
                    $expire_date = $start_date->copy()->addYears($value);
                }
            }
        }


        // $start_date = now()->format('Y-m-d');
        // $start_date = $start_date->format('Y-m-d');
        $expire_date = $expire_date->format('Y-m-d');
        $start_date_do = $start_date->format('Y-m-d');

        $check_user_subscription = UserSubscription::where('user_id', $user_id)->first();

        if ($check_user_subscription) {
            // Update the existing subscription and reset status to 0
            $check_user_subscription->update([
                'plan_name' => $plan_name,
                'price' => $price,
                'payment_mode' => $payment_mode,
                'start_date' => $start_date_do,
                'expire_date' => $expire_date,
                'status' => 0, // reset status
                'subscription_id' => $subscription_id,
            ]);
        } else {
            // Create new subscription if not found
            $check_user_subscription = UserSubscription::create([
                'user_id' => $user_id,
                'plan_name' => $plan_name,
                'price' => $price,
                'payment_mode' => $payment_mode,
                'start_date' => $start_date_do,
                'expire_date' => $expire_date,
                'status' => 0,
                'subscription_id' => $subscription_id,
            ]);
        }

        UserSubscriptionHistory::create([

            'user_id' => $user_id,
            'plan_name' => $plan_name,
            'price' => $price,
            'payment_mode' => $payment_mode,
            'start_date' => $start_date_do,
            'expire_date' => $expire_date,
            'status' => 0,
            'subscription_id' => $subscription_id,

        ]);

        $FcmToken = User::select('device_token')->where('id', $user_id)->first()->device_token;

        $data = [
            'title' => "Subcription Succesfull",
            'message' => "you have successfully subscribed to $plan_name",
            'type' => "subscriber",
            'booking_id' => 0,
            'order_id' => 0,
        ];
        $baseController = new BaseController();
        $baseController->sendNotification(new Request($data), $FcmToken);

        // dd($data);

        // Check if the subscription was successfully created or updated
        if ($check_user_subscription) {
            return response()->json([
                'status' => true,
                'message' => 'Payment successful, subscription done',
                'subscription' => [
                    'id' => $check_user_subscription->id,
                    'user_id' => (int)$check_user_subscription->user_id,
                    'plan_name' => $check_user_subscription->plan_name ?? '',
                    'price' => $check_user_subscription->price ?? '',
                    'start_date' => $check_user_subscription->start_date ?? '',
                    'expire_date' => $check_user_subscription->expire_date ?? '',
                    'payment_mode' => $check_user_subscription->payment_mode ?? '',
                    'status' => $check_user_subscription->status ?? 0,
                    'created_at' => $check_user_subscription->created_at,
                    'updated_at' => $check_user_subscription->updated_at,
                    'subscription_id' => $check_user_subscription->subscription_id ?? '',
                ]
            ], 200);
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Payment failed, subscription could not be processed',
            ], 500);
        }
    }


    // searchVendorServices
    public function searchVendorServices(Request $request)
    {
        // $vendor_id = $request->vendor_id;
        $user = Auth::user();
        $vendor_id = $user->id;
        $service_name = $request->service_name;

        // Check if vendor_id is provided
        // if (!$vendor_id) {
        //     return response()->json([
        //         'status' => false,
        //         'message' => 'Vendor ID is required',
        //     ]);
        // }

        // Check if service_name is provided
        if ($service_name != '') {
            // Query services filtered by both vendor_id and service_name (strict check)
            $services = Service::select(
                'services.id',
                'services.category_id',
                'services.service_name',
                'services.address',
                'categories.category_name',
                'services.created_at'
            )
                ->join('categories', 'services.category_id', '=', 'categories.id')
                ->where('services.vendor_id', $vendor_id) // Ensure vendor_id matches
                ->where('services.service_name', 'LIKE', substr($service_name, 0, 3) . '%') // Ensure service_name matches the first 3 characters
                ->orderBy('services.created_at', 'desc')
                ->get();

            // If no services found for the vendor, return appropriate message
            if ($services->isEmpty()) {
                return response()->json([
                    'status' => false,
                    'message' => 'The service you searched does not belong to this vendor',
                    'serviceSearch' => [],
                ]);
            }

            $serviceSearch = [];

            // Loop through each service and gather details
            foreach ($services as $service) {
                $serviceDetail = [];
                $serviceDetail['id'] = $service->id;
                $serviceDetail['service_name'] = $service->service_name ?? '';
                $serviceDetail['category_name'] = $service->category_name ?? '';
                $serviceDetail['address'] = $service->address ?? '';

                // Fetch service images
                $serviceImages = ServiceImages::where('service_id', $service->id)->get();
                $serviceDetail['service_images'] = $serviceImages->isNotEmpty()
                    ? $serviceImages->map(function ($image) {
                        return url('/assets/images/service_images/' . $image->service_images);
                    })
                    : [url('/assets/images/service_images/default_service.png')];

                // Fetch and calculate total reviews and average rating
                $productReviews = ServiceReview::where('service_id', $service->id)->get();
                $totalReviewCount = $productReviews->count();
                $totalReviewStar = $productReviews->sum('review_star');
                $totalAvgReview = $totalReviewCount > 0 ? (string) round($totalReviewStar / $totalReviewCount, 1) : '';

                $serviceDetail['totalReviewCount'] = $totalReviewCount;
                $serviceDetail['totalAvgReview'] = $totalAvgReview;

                $serviceSearch[] = $serviceDetail;
            }

            return response()->json([
                'status' => true,
                'message' => 'Services Search',
                'serviceSearch' => $serviceSearch,
            ]);
        } else {
            // If service_name is not provided, return error
            return response()->json([
                'status' => false,
                'message' => 'Service Name is required',
            ]);
        }
    }


    // removeServiceImages
    public function removeServiceImages(Request $request)
    {
        // Get the authenticated vendor's ID
        // $vendor_id = $request->vendor_id;
        $user = Auth::user();
        $vendor_id = $user->id;

        // Check if the authenticated vendor ID is available
        if ($vendor_id) {
            // Validate the request to ensure product_id and product_image_id are provided
            $request->validate([
                'service_id' => 'required|exists:services,id',
                'service_image_id' => 'required|exists:service_images,id',
            ]);

            // Check if the product exists and belongs to the authenticated vendor
            $productExists = Service::where('id', $request->service_id)
                ->where('vendor_id', $vendor_id)
                ->exists();

            if ($productExists) {
                // Verify that the image belongs to the product and vendor by checking product_images table
                $productImage = ServiceImages::where('id', $request->service_image_id)
                    ->where('service_id', $request->service_id)
                    ->first();

                // Check if the product image exists
                if ($productImage) {
                    // Remove the product image
                    $productImage->delete();

                    return response()->json([
                        'success' => true,
                        'message' => 'Service image removed successfully!',
                    ], 200);
                } else {
                    return response()->json([
                        'success' => false,
                        'message' => 'Service image not found or it does not belong to this product.',
                    ], 404);
                }
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Service not found or you do not have permission to delete images for this product.',
                ], 404);
            }
        } else {
            return response()->json([
                'success' => false,
                'message' => 'Vendor not authenticated.',
            ], 401);
        }
    }


    // totalPercentage
    // totalPercentage
    public function totalPercentage(Request $request)
    {
        // Get the authenticated vendor's ID
        $user = Auth::user();
        $vendor_id = $user->id;
        // $vendor_id = $request->vendor_id;

        // Check if the vendor ID is available
        if (!$vendor_id) {
            return response()->json([
                'status' => false,
                'message' => 'Vendor not authenticated.',
            ], 401);
        }

        // Retrieve the vendor's data from the users table
        $vendor = User::find($vendor_id);

        if (!$vendor) {
            return response()->json([
                'status' => false,
                'message' => 'Vendor not found.',
            ], 404);
        }

        // Initialize profile completion percentage for user fields
        $userFields = ['first_name', 'last_name', 'email', 'mobile', 'image'];
        $filledUserFields = 0;

        // Check each user field and count filled ones
        foreach ($userFields as $field) {
            if (!empty($vendor->$field)) {
                $filledUserFields++;
            }
        }

        // Retrieve the first service associated with the vendor
        $service = Service::where('vendor_id', $vendor_id)->first();

        // Initialize profile completion percentage for service fields
        $serviceFields = [
            'category_id',
            'subcategory_id',
            'service_name',
            'service_description',
            'service_website',
            'service_phone',
            'service_email',
            'address',
            'lat',
            'lon',
            'area',
            'city',
            'state',
            'open_days',
            'open_time',
            'close_time',
            'published_month',
            'published_year',
            'employee_strength',
            'cover_image',
            'meta_title',
            'meta_description',
            'video_url',
            'instagram_link',
            'facebook_link',
            'whatsapp_link',
            'twitter_link'
        ];
        $filledServiceFields = 0;

        // Check each service field and count filled ones if service exists
        if ($service) {
            foreach ($serviceFields as $field) {
                if (!empty($service->$field)) {
                    $filledServiceFields++;
                }
            }

            // Check if there is at least one service image
            $hasServiceImage = ServiceImages::where('service_id', $service->id)->exists();
            if ($hasServiceImage) {
                $filledServiceFields++; // Count as one additional completed field
            }

            // Check if there is at least one store with this service_id
            $hasStore = Stores::where('service_id', $service->id)->exists();
            if ($hasStore) {
                $filledServiceFields++; // Count as one additional completed field
            }
        }

        // Calculate the total profile completion percentage
        $totalFields = count($userFields) + count($serviceFields) + 2; // +1 for service images, +1 for store
        $filledFields = $filledUserFields + $filledServiceFields;
        $percentage = round(($filledFields / $totalFields) * 100); // Round the percentage

        return response()->json([
            'status' => true,
            'percentage' => $percentage,
        ]);
    }



    //addStore
    public function addStore(Request $request)
    {
        // $vendor_id = $request->vendor_id;
        $user = Auth::user();
        $vendor_id = $user->id;

        // if (!$vendor_id) {
        //     return response()->json(
        //         [
        //             'status' => false,
        //             'message' => 'Vendor ID is required',
        //         ],
        //         400
        //     );
        // }

        $rules = [
            'store_name' => 'required',
            'store_description' => 'required',
            'service_id' => 'required|exists:services,id',
            'price' => 'required',
        ];

        $customMessages = [
            'store_name.required' => 'Please enter the service name.',
            'store_description.required' => 'Please enter the service description.',
            'service_id.required' => 'Please select a store.',
            'service_id.exists' => 'The selected store is invalid.',
            'price' => 'Please enter the service price.',
        ];


        $validator = Validator::make($request->all(), $rules, $customMessages);
        if ($validator->fails()) {
            $errors = $validator->errors();

            $responseErrors = [];
            foreach ($errors->all() as $error) {
                $responseErrors = $error;
            }

            return response(
                [
                    'status' => false,
                    'message' => $responseErrors,
                ],
                401
            );
        }


        $data = new Stores();
        $data->service_id = $request->service_id;
        $data->store_name = $request->store_name;
        $data->store_description = $request->store_description;
        $data->price = $request->price;


        $data->save();

        // Handle store images
        if ($request->hasFile('store_images')) {
            foreach ($request->file('store_images') as $file) {
                $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                $file->move(public_path('assets/images/store_images'), $fileName);

                $data->storeImages()->create([
                    'store_id' => $data->id,
                    'store_images' => $fileName,
                ]);
            }
        }

        // Handle store attachments
        if ($request->hasFile('store_attachments')) {
            foreach ($request->file('store_attachments') as $file) {
                $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                $file->move(public_path('assets/images/store_attachments'), $fileName);

                $data->storeAttachments()->create([
                    'store_id' => $data->id,
                    'store_attachments' => $fileName,
                ]);
            }
        }


        return response(
            [
                'status' => true,
                'message' => 'Service added successfully...!',
                'service_id' => $data->id,
            ],
            200
        );
    }


    // storeList
    public function storeList(Request $request)
    {
        $service_id = $request->service_id;

        // Validate the `service_id` parameter
        if (!$service_id) {
            return response()->json([
                'status' => false,
                'message' => 'Service ID is required',
            ], 400);
        }

        // Fetch vendor_id and service_name from the services table
        $service = Service::select('id', 'vendor_id', 'service_name')->where('id', $service_id)->first();
        if (!$service) {
            return response()->json([
                'status' => false,
                'message' => 'Invalid Service ID',
            ], 400);
        }
        $vendor_id = $service->vendor_id;

        // Fetch vendor details from the users table
        $vendor = User::select('first_name', 'last_name', 'email', 'image')
            ->where('id', $vendor_id)
            ->first();

        // Ensure vendor details are not null
        $vendor_details = $vendor ? [
            'first_name' => $vendor->first_name ?: '',
            'last_name' => $vendor->last_name ?: '',
            'email' => $vendor->email ?: '',
            'image' => $vendor->image ? url('assets/images/users_images/' . $vendor->image) : url('assets/images/users_images/default_user.jpg'),
        ] : [];

        // Fetch store data based on `service_id`
        $data = Stores::select(
            'stores.id',
            'stores.store_name',
            'stores.store_description',
            'stores.price',
            'stores.subcategory_id' // Fetch subcategory_id
        )
            ->where('service_id', $service_id) // Filter by service_id
            ->orderBy('created_at', 'desc')
            ->get()
            ->transform(function ($store) use ($vendor_details) {
                // Fetch all store images, ordered by latest first
                $images = StoreImages::where('store_id', $store->id)
                    ->orderBy('created_at', 'desc')
                    ->get();
                $store->store_images = $images->isNotEmpty()
                    ? $images->map(function ($image) {
                        return [
                            'id' => $image->id,
                            'url' => url('assets/images/store_images/' . $image->store_images),
                        ];
                    })->toArray()
                    : [];

                // Fetch all store attachments, ordered by latest first
                $attachments = StoreAttachments::where('store_id', $store->id)
                    ->orderBy('created_at', 'desc')
                    ->get();
                $store->store_attachments = $attachments->isNotEmpty()
                    ? $attachments->map(function ($attachment) {
                        return [
                            'id' => $attachment->id,
                            'url' => url('assets/images/store_attachments/' . $attachment->store_images),
                        ];
                    })->toArray()
                    : [];

                // Fetch subcategory name using subcategory_id
                $subcategory = Subcategory::select('subcategory_name')
                    ->where('id', $store->subcategory_id)
                    ->first();

                // Ensure store data fields are not null
                $store->store_name = $store->store_name ?: '';
                $store->store_description = $store->store_description ?: '';
                $store->price = $store->price ?: '';
                $store->subcategory_id = $store->subcategory_id ?: 0;

                // Attach subcategory name
                $store->subcategory_name = $subcategory ? $subcategory->subcategory_name : '';

                // Attach vendor details to each store
                $store->vendor_details = $vendor_details;

                return $store;
            });

        // Return response with service_name on top
        return response()->json([
            'status' => $data->isNotEmpty(),
            'message' => $data->isNotEmpty() ? 'Store List found' : 'No services found add your service',
            'service_name' => $service->service_name, // Added service name in response
            'ServiceList' => $data,
        ]);
    }



    // updateStore
    public function updateStore(Request $request)
    {
        $user = Auth::user();
        $vendor_id = $user->id;
        try {

            $store_id = $request->store_id;

            // Validate store_id parameter
            if (!$store_id) {
                return response([
                    'status' => false,
                    'message' => 'Store ID is required',
                ], 400);
            }

            // Find the store by store_id
            $store = Stores::find($store_id);

            if (!$store) {
                return response([
                    'status' => false,
                    'message' => 'Store not found',
                ], 404);
            }

            // Update the store details
            $store->update($request->only([
                'store_name',
                'store_description',
                'price',
                'subcategory_id',
            ]));

            // Handle store images
            if ($request->hasFile('store_images')) {
                foreach ($request->file('store_images') as $file) {
                    $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                    $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                    $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                    $file->move(public_path('assets/images/store_images'), $fileName);

                    // Save image record in the store_images table
                    $store->storeImages()->create([
                        'store_images' => $fileName,
                    ]);
                }
            }

            // Handle store attachments
            if ($request->hasFile('store_attachments')) {
                foreach ($request->file('store_attachments') as $file) {
                    $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                    $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                    $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                    $file->move(public_path('assets/images/store_attachments'), $fileName);

                    // Save attachment record in the store_attachments table
                    $store->storeAttachments()->create([
                        'store_attachments' => $fileName,
                    ]);
                }
            }

            // Fetch subcategory name
            $subcategory = SubCategory::find($store->subcategory_id);
            $subcategoryName = $subcategory ? $subcategory->subcategory_name : null;

            // Format store data with full image and attachment paths
            $storeData = $store->only(['id', 'service_id', 'store_name', 'store_description', 'subcategory_id', 'price', 'created_at', 'updated_at']);
            $storeData['subcategory_name'] = $subcategoryName;


            // Add store images
            $storeData['storeImages'] = $store->storeImages->map(function ($image) {
                return [
                    'id' => $image->id,
                    'store_images' => asset('assets/images/store_images/' . $image->store_images),
                ];
            });

            // Add store attachments
            $storeData['storeAttachments'] = $store->storeAttachments->map(function ($attachment) {
                return [
                    'id' => $attachment->id,
                    'store_attachments' => asset('assets/images/store_attachments/' . $attachment->store_attachments),
                ];
            });

            return response([
                'status' => true,
                'message' => 'Store updated successfully!',
                'store' => $storeData,
            ], 200);
        } catch (\Exception $e) {
            return response([
                'status' => false,
                'message' => 'An error occurred while updating the store: ' . $e->getMessage(),
            ], 500);
        }
    }



    // removeStoreImages
    public function removeStoreImages(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            // Validate the request to ensure store_id and store_image_id are provided
            $request->validate([
                'store_id' => 'required|exists:stores,id', // Ensure the store exists
                'store_image_id' => 'required|exists:store_images,id', // Ensure the image exists
            ]);

            // Check if the store exists
            $store = Stores::find($request->store_id);

            if (!$store) {
                return response()->json([
                    'success' => false,
                    'message' => 'Store not found.',
                ], 404);
            }

            // Verify that the image belongs to the store
            $storeImage = StoreImages::where('id', $request->store_image_id)
                ->where('store_id', $request->store_id)
                ->first();

            if (!$storeImage) {
                return response()->json([
                    'success' => false,
                    'message' => 'Store image not found or does not belong to the specified store.',
                ], 404);
            }

            // Remove the image file from the server
            $imagePath = public_path('assets/images/store_images/' . $storeImage->store_images);
            if (file_exists($imagePath)) {
                unlink($imagePath); // Delete the file from the server
            }

            // Delete the image record from the database
            $storeImage->delete();

            return response()->json([
                'success' => true,
                'message' => 'Store image removed successfully!',
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while removing the store image: ' . $e->getMessage(),
            ], 500);
        }
    }


    // removeStoreAttachment
    public function removeStoreAttachment(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            // Validate the request to ensure store_id and store_image_id are provided
            $request->validate([
                'store_id' => 'required|exists:stores,id', // Ensure the store exists
                'store_attach_id' => 'required|exists:store_attachments,id', // Ensure the image exists
            ]);

            // Check if the store exists
            $store = Stores::find($request->store_id);

            if (!$store) {
                return response()->json([
                    'success' => false,
                    'message' => 'Store not found.',
                ], 404);
            }

            // Verify that the image belongs to the store
            $storeImage = StoreAttachments::where('id', $request->store_attach_id)
                ->where('store_id', $request->store_id)
                ->first();

            if (!$storeImage) {
                return response()->json([
                    'success' => false,
                    'message' => 'Store image not found or does not belong to the specified store.',
                ], 404);
            }

            // Remove the image file from the server
            $imagePath = public_path('assets/images/store_attachments/' . $storeImage->store_attachments);
            if (file_exists($imagePath)) {
                unlink($imagePath); // Delete the file from the server
            }

            // Delete the image record from the database
            $storeImage->delete();

            return response()->json([
                'success' => true,
                'message' => 'Store attachment removed successfully!',
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while removing the store image: ' . $e->getMessage(),
            ], 500);
        }
    }


    // userOnline
    public function userOnline(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {

            $validated = $request->validate([
                'status' => 'required',
                // 'user_id' => 'required|exists:users,id',
            ]);

            // Extract the parameters from the request
            $status = $validated['status'];
            // $user_id = $validated['user_id'];

            // Assuming you are updating the user's status in the database
            $user = User::find($user_id);
            if ($user) {
                $user->status = $status;
                $user->save();
            }

            return response([
                'status' => true,
                'message' => 'User status updated successfully!',
                'is_online' => $user->status,
            ], 200);
        } catch (\Exception $e) {
            return response([
                'status' => false,
                'message' => 'An error occurred while updating the status: ' . $e->getMessage(),
                'is_online' => $user->status,
            ], 500);
        }
    }


    // vendorInfo
    public function vendorInfo(Request $request)
    {
        $user = Auth::user();
        $vendor_id = $user->id;
        // $vendor_id = $request->vendor_id;
        $user_id = $request->user_id; // Get user_id from request

        // Fetch vendor details from the users table
        $data = User::where('id', $vendor_id)->first();

        if (!$data) {
            return response()->json([
                'status' => false,
                'message' => 'Vendor not found ...!',
            ], 404);
        } else {
            // Calculate total reviews and average rating
            $serviceIds = Service::where('vendor_id', $vendor_id)->pluck('id');
            $productReviews = ServiceReview::whereIn('service_id', $serviceIds)->get();

            $totalReviewCount = $productReviews->count();
            $totalReviewStar = $productReviews->sum('review_star');

            // Ensure it always returns a string formatted to one decimal place
            $totalAvgReview = $totalReviewCount > 0 ? number_format($totalReviewStar / $totalReviewCount, 1) : '0.0';

            // Fetch services belonging to the vendor
            $serviceDetails = Service::where('vendor_id', $vendor_id)->get()->map(function ($service) use ($user_id) {
                // Fetch service images
                $serviceImages = ServiceImages::where('service_id', $service->id)->get();
                $service_images = $serviceImages->isNotEmpty()
                    ? $serviceImages->map(fn($image) => url('assets/images/service_images/' . $image->service_images))
                    : [url('assets/images/service_images/default_service.png')];

                // Fetch category name from categories table
                $category = Category::select('category_name')->where('id', $service->category_id)->first();
                $categoryName = $category ? $category->category_name : '';

                // Fetch default currency
                $defaultCurrency = SiteSetup::first()->default_currency;

                // Calculate price range
                $storePrices = Stores::where('service_id', $service->id)->pluck('price');
                $minPrice = $storePrices->isNotEmpty() ? $storePrices->min() : 0;
                $maxPrice = $storePrices->isNotEmpty() ? $storePrices->max() : 0;
                $priceRange = $minPrice === $maxPrice
                    ? "$defaultCurrency$minPrice"
                    : "$defaultCurrency$minPrice - $defaultCurrency$maxPrice";

                // Fetch vendor details
                $user = User::where('id', $service->vendor_id)->first();
                $vendorFirstName = $user ? $user->first_name : "";
                $vendorLastName = $user ? $user->last_name : "";
                $vendorEmail = $user ? $user->email : "";
                $vendorImage = $user ? url('assets/images/users_images/' . $user->image) : "";

                // Fetch the like status
                $isLiked = ServiceLike::where('service_id', $service->id)
                    ->where('user_id', $user_id)
                    ->exists();
                $isLike = $isLiked ? 1 : 0;

                // Replace `null` values with empty strings and format the service details
                return [
                    'id' => $service->id,
                    'vendor_id' => $service->vendor_id,
                    'category_id' => $service->category_id ?? 0,
                    'category_name' => $categoryName,
                    'subcategory_id' => $service->subcategory_id ?? '',
                    'service_name' => $service->service_name ?? '',
                    'address' => $service->address ?? '',
                    'lat' => $service->lat ?? '',
                    'lon' => $service->lon ?? '',
                    'is_featured' => $service->is_featured ?? 0,
                    'published_month' => $service->published_month ?? '',
                    'published_year' => $service->published_year ?? '',
                    'service_images' => $service_images,
                    'total_reviews' => $serviceReviews = ServiceReview::where('service_id', $service->id)->count(),
                    'average_rating' => $serviceReviews > 0
                        ? number_format(ServiceReview::where('service_id', $service->id)->sum('review_star') / $serviceReviews, 1)
                        : '0.0',
                    'price_range' => $priceRange,
                    'vendor_first_name' => $vendorFirstName,
                    'vendor_last_name' => $vendorLastName,
                    'vendor_email' => $vendorEmail,
                    'vendor_image' => $vendorImage,
                    'isLike' => $isLike,  // Added like status
                ];
            });

            // If the vendor has services, get the first service's lat and lon
            $serviceLatLon = Service::where('vendor_id', $vendor_id)->first(['lat', 'lon', 'address']);

            // Vendor details
            $userdata = [
                'id' => $data->id,
                'first_name' => $data->first_name ?? '',
                'last_name' => $data->last_name ?? '',
                'address' => $serviceLatLon->address ?? '',
                'lat' => $serviceLatLon->lat ?? '', // Service latitude
                'lon' => $serviceLatLon->lon ?? '', // Service longitude
                'mobile' => $data->mobile ?? '',
                'email' => $data->email ?? '',
                'country_code' => $data->country_code ?? '',
                'image' => $data->image
                    ? url('assets/images/users_images/' . $data->image)
                    : url('assets/images/users_images/default_user.jpg'),
                'total_reviews' => $totalReviewCount,
                'average_rating' => $totalAvgReview
            ];

            return response()->json([
                'status' => true,
                'message' => 'Vendor details found',
                'Vendordetails' => $userdata,
                'serviceDetails' => $serviceDetails
            ], 200);
        }
    }








    // deleteStore
    public function deleteStore(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        $request->validate([
            'store_id' => 'required|exists:stores,id'
        ]);

        $store_id = $request->store_id;

        // Check if the store exists
        $store = Stores::find($store_id);
        if (!$store) {
            return response()->json([
                'status' => false,
                'message' => 'Service not found',
            ], 404);
        }

        // Delete related data
        StoreAttachments::where('store_id', $store_id)->delete();
        StoreImages::where('store_id', $store_id)->delete();
        $store->delete();

        return response()->json([
            'status' => true,
            'message' => 'Service deleted successfully',
        ], 200);
    }








    // addCampaign
    public function addCampaign(Request $request)
    {
        $user = Auth::user();
        $vendor_id = $user->id;
        $rules = [
            'campaign_name' => 'required',
            // 'vendor_id' => 'required|exists:users,id',
            'service_id' => 'required|exists:services,id',
            'address' => 'required',
            'lat' => 'required',
            'lon' => 'required',
            'area_distance' => 'required',
        ];

        $customMessages = [
            'campaign_name.required' => 'Please enter the campaign name.',
            // 'vendor_id.required' => 'Vendor ID is required.',
            // 'vendor_id.exists' => 'Vendor ID does not exist.',
            'service_id.required' => 'Service ID is required.',
            'service_id.exists' => 'Service ID does not exist.',
            'address.required' => 'Please enter the address.',
            'lat.required' => 'Please enter the latitude.',
            'lon.required' => 'Please enter the longitude.',
            'area_distance.required' => 'Please enter the area distance.',
        ];

        $validator = Validator::make($request->all(), $rules, $customMessages);
        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'message' => $validator->errors()->first(),
            ], 400);
        }

        $campaign = new Campaign();
        $campaign->vendor_id = $vendor_id;
        $campaign->service_id = $request->service_id;
        $campaign->campaign_name = $request->campaign_name;
        $campaign->address = $request->address;
        $campaign->lat = $request->lat;
        $campaign->lon = $request->lon;
        $campaign->area_distance = $request->area_distance;
        $campaign->save();

        return response()->json([
            'status' => true,
            'message' => 'Campaign added successfully!',
            'campaign' => $campaign,
        ], 200);
    }

    // addCampaign
    // public function addCampaign(Request $request)
    // {
    //     $rules = [
    //         'campaign_name' => 'required',
    //         'vendor_id' => 'required|exists:users,id',
    //         'service_id' => 'required|exists:services,id',
    //         'address' => 'required',
    //         'lat' => 'required',
    //         'lon' => 'required',
    //         'area_distance' => 'required',
    //     ];

    //     $customMessages = [
    //         'campaign_name.required' => 'Please enter the campaign name.',
    //         'vendor_id.required' => 'Vendor ID is required.',
    //         'vendor_id.exists' => 'Vendor ID does not exist.',
    //         'service_id.required' => 'Service ID is required.',
    //         'service_id.exists' => 'Service ID does not exist.',
    //         'address.required' => 'Please enter the address.',
    //         'lat.required' => 'Please enter the latitude.',
    //         'lon.required' => 'Please enter the longitude.',
    //         'area_distance.required' => 'Please enter the area distance.',
    //     ];

    //     $validator = Validator::make($request->all(), $rules, $customMessages);
    //     if ($validator->fails()) {
    //         return response()->json([
    //             'status' => false,
    //             'message' => $validator->errors()->first(),
    //         ], 400);
    //     }

    //     // Fetch the latest campaign for the given service_id
    //     $latestCampaign = DB::table('campaign')
    //         ->where('service_id', $request->service_id)
    //         ->orderBy('id', 'desc')
    //         ->first();

    //     if ($latestCampaign) {
    //         // Fetch the latest goal_payment entry for the campaign
    //         $latestGoalPayment = DB::table('goal_payment')
    //             ->where('campaign_id', $latestCampaign->id)
    //             ->orderBy('id', 'desc')
    //             ->first();

    //         if ($latestGoalPayment) {
    //             $endDate = Carbon::parse($latestGoalPayment->end_date); // Convert to Carbon instance
    //             $currentDate = Carbon::today(); // Get today's date

    //             if ($endDate->greaterThanOrEqualTo($currentDate)) {
    //                 return response()->json([
    //                     'status' => false,
    //                     'message' => 'A sponsored campaign for this service is already active. Please wait until it expires before adding a new one.',
    //                 ], 400);
    //             } else {
    //                 return response()->json([
    //                     'status' => false,
    //                     'message' => 'The sponsored plan for this service has expired. Please activate a new one.',
    //                 ], 400);
    //             }
    //         }
    //     }

    //     // Create and save campaign
    //     $campaign = new Campaign();
    //     $campaign->vendor_id = $request->vendor_id;
    //     $campaign->service_id = $request->service_id;
    //     $campaign->campaign_name = $request->campaign_name;
    //     $campaign->address = $request->address;
    //     $campaign->lat = $request->lat;
    //     $campaign->lon = $request->lon;
    //     $campaign->area_distance = $request->area_distance;
    //     $campaign->save();

    //     return response()->json([
    //         'status' => true,
    //         'message' => 'Campaign added successfully!',
    //         'campaign' => $campaign,
    //     ], 200);
    // }







    // addGoals
    public function addGoals(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $rules = [
            'start_date'   => 'required',
            'campaign_id'  => 'required|exists:campaign,id',
            'end_date'     => 'required',
            'days'         => 'required',
            'price'        => 'required',
        ];

        $customMessages = [
            'start_date.required'  => 'Please enter the start date.',
            'end_date.required'    => 'Please enter the end date.',
            'days.required'        => 'Please enter the number of days.',
            'price.required'       => 'Please enter the price.',
            'campaign_id.exists'   => 'The selected campaign does not exist.',
        ];

        $validator = Validator::make($request->all(), $rules, $customMessages);

        if ($validator->fails()) {
            return response()->json([
                'status'  => false,
                'message' => $validator->errors()->first(),
            ], 400);
        }

        // Fetch campaign details
        $campaign = DB::table('campaign')
            ->where('id', $request->campaign_id)
            ->select('id', 'campaign_name', 'address', 'lat', 'lon', 'area_distance')
            ->first();

        if (!$campaign) {
            return response()->json([
                'status'  => false,
                'message' => 'Campaign not found.',
            ], 400);
        }

        // Fetch default currency
        $defaultCurrency = SiteSetup::first()->default_currency ?? '$';

        $goal = new Goals();
        $goal->start_date = $request->start_date;
        $goal->campaign_id = $request->campaign_id;
        $goal->end_date = $request->end_date;
        $goal->days = $request->days;
        $goal->price = $request->price;
        $goal->save();

        return response()->json([
            'status'  => true,
            'message' => 'Goal added successfully!',
            'goal'    => [
                'id'        => $goal->id,
                'start_date' => $goal->start_date,
                'end_date'  => $goal->end_date,
                'days'      => $goal->days,
                'price'     => $defaultCurrency . $goal->price, // Currency before price
            ],
            'campaign_details' => $campaign,
        ], 200);
    }


    public function getBudgetCount(Request $request)
    {
        $budgetData = [];

        for ($i = 1; $i <= 30; $i++) {
            $dayLabel = $i === 1 ? 'Day' : 'Days';
            $budgetData[] = [
                'id' => $i,
                'days' => "$i $dayLabel",
                'price' => $i * 100
            ];
        }

        return response()->json([
            'status' => true,
            'message' => 'Budget data retrieved successfully',
            'data' => $budgetData
        ]);
    }



    public function goalPaymentSuccess(Request $request)
    {
        $user = Auth::user();
        $vendor_id = $user->id;
        $goal_id = $request->input('goal_id');
        // $vendor_id = $request->input('vendor_id');
        $payment_mode = $request->input('payment_mode');
        $service_id = $request->input('service_id');
        $price = $request->input('price');

        // Validate input data
        if (!$goal_id || !$payment_mode || !$price || !$service_id) {
            return response()->json([
                'status' => false,
                'message' => 'Invalid input data',
            ], 400);
        }

        // Check if vendor exists in users table
        $vendorExists = User::where('id', $vendor_id)->exists();
        if (!$vendorExists) {
            return response()->json([
                'status' => false,
                'message' => 'Vendor not found',
            ], 404);
        }

        // Check if service exists in services table
        $serviceExists = Service::where('id', $service_id)->exists();
        if (!$serviceExists) {
            return response()->json([
                'status' => false,
                'message' => 'Service not found',
            ], 404);
        }

        // Check if goal exists
        $goal_payment = Goals::where('id', $goal_id)->first();
        if (!$goal_payment) {
            return response()->json([
                'status' => false,
                'message' => 'Goal not found',
            ], 404);
        }

        $end_date = $goal_payment->end_date;
        $campaign_id = $goal_payment->campaign_id;

        $formatted_date = Carbon::createFromFormat('d-m-Y', $end_date)->format('Y-m-d');


        // dd($formatted_date);


        try {
            // Check if a record with the same goal_id and service_id exists
            $payment = GoalPayment::where('service_id', $service_id)->first();

            if ($payment) {
                // Update existing record
                $payment->vendor_id = $vendor_id;
                $payment->goal_id = $goal_id;
                $payment->payment_mode = $payment_mode;
                $payment->price = $price;
                $payment->end_date = $end_date;
                $payment->campaign_id = $campaign_id;
                $payment->end_date_do = $formatted_date;
                $payment->save();
            } else {
                // Insert new record
                $newPayment = new GoalPayment();
                $newPayment->goal_id = $goal_id;
                $newPayment->vendor_id = $vendor_id;
                $newPayment->payment_mode = $payment_mode;
                $newPayment->service_id = $service_id;
                $newPayment->price = $price;
                $newPayment->end_date = $end_date;
                $newPayment->campaign_id = $campaign_id;
                $newPayment->end_date_do = $formatted_date;
                $newPayment->save();
            }

            GoalPaymentHistory::create([
                'vendor_id' => $vendor_id,
                'goal_id' => $goal_id,
                'payment_mode' => $payment_mode,
                'price' => $price,
                'end_date' => $end_date,
                'campaign_id' => $campaign_id,
                'service_id' => $service_id,
                'end_date_do' => $formatted_date,
            ]);

            // Fetch the updated goal data
            $goalData = Goals::where('id', $goal_id)->first();

            return response()->json([
                'status' => true,
                'message' => 'Payment successful',
                'goal_data' => $goalData, // Returning complete goal data
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Payment failed: ' . $e->getMessage(),
            ], 500);
        }
    }


    // getCampaign
    public function getCampaign(Request $request)
    {

        $user = Auth::user();
        $vendor_id = $user->id;
        if (!empty($vendor_id)) {
            // Fetch campaigns related to the vendor
            $campaigns = Campaign::where('vendor_id', $request->vendor_id)->get();

            $isPayment = GoalPayment::where('vendor_id', $request->vendor_id)->first();



            if ($isPayment) {

                $is_expiary = $isPayment->end_date_do;

                $time = now();



                if ($is_expiary >= $time) {

                    $is_payment_status = "1";
                } else {

                    $is_payment_status = "0";
                }
            } else {

                $is_payment_status = "0"; // Explicitly set to "0" instead of null

            }

            if ($campaigns->isNotEmpty()) {
                foreach ($campaigns as $campaign) {
                    $campaign->campaign_name = $campaign->campaign_name ?? "";
                    $campaign->address = $campaign->address ?? "";
                    $campaign->lat = $campaign->lat ?? "";
                    $campaign->lon = $campaign->lon ?? "";
                    $campaign->area_distance = $campaign->area_distance ?? "";

                    // Fetch goals related to this campaign
                    $campaign->goals = DB::table('goals')
                        ->where('campaign_id', $campaign->id)
                        ->get();
                }

                return response()->json([
                    'status' => true,
                    'message' => 'Campaigns Found',
                    'campaignData' => $campaigns,
                    'is_payment_status' => $is_payment_status,
                ]);
            } else {
                return response()->json([
                    'status' => false,
                    'message' => 'No campaigns found for this vendor',
                    'campaignData' => [],
                ]);
            }
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Please provide a vendor_id',
                'campaignData' => [],
            ]);
        }
    }







    // // getServiceInsights
    public function getServiceInsights(Request $request)
    {
        // Validate inputs
        $request->validate([
            'service_id' => 'required|exists:services,id', // Ensure the service_id exists in the services table
            'monthname' => 'required|in:January,February,March,April,May,June,July,August,September,October,November,December', // Validate full month names
        ]);

        $service_id = $request->service_id;
        $monthname = $request->monthname;

        // Convert full month name to a numeric month (e.g., March => 03)
        $month = date('m', strtotime($monthname . ' 1')); // Ensures proper parsing
        $year = date('Y'); // Current year

        // Get the number of days in the selected month
        $daysInMonth = cal_days_in_month(CAL_GREGORIAN, $month, $year);

        // Fetch user visits from the ServiceCount table based on service_id and month
        $data = ServiceCount::selectRaw('DAY(created_at) as day, COUNT(*) as user_visits')
            ->where('service_id', $service_id)
            ->whereMonth('created_at', $month)
            ->whereYear('created_at', $year)
            ->groupByRaw('DAY(created_at)')
            ->orderByRaw('DAY(created_at)')
            ->get()
            ->keyBy('day'); // Key by day for easy lookup

        // Fetch total store likes from ServiceLike model
        $storeLikes = ServiceLike::where('service_id', $service_id)->count();

        // Fetch total leads from ServiceLead model
        $totalLeads = ServiceLead::where('service_id', $service_id)->count();

        // Build a complete month dataset with all days
        $graphdata = [];
        $totalUserVisits = 0;

        for ($day = 1; $day <= $daysInMonth; $day++) {
            $user_visits = $data[$day]->user_visits ?? 0; // If no data for the day, set to 0
            $totalUserVisits += $user_visits;

            $graphdata[] = [
                'date' => $day . ' ' . $monthname, // Format: 1 March, 2 March, etc.
                'user_visits' => $user_visits,
            ];
        }

        // Return the response including store likes and leads
        return response()->json([
            'status' => true,
            'message' => 'Service insights fetched successfully.',
            'CountRetrieved' => [
                'storevisits' => $totalUserVisits, // Total visits for the full month
                'storelikes' => $storeLikes, // Total store likes
                'leads' => $totalLeads, // Total service leads
            ],
            'graphdata' => $graphdata, // Updated object name
        ], 200);
    }





    // getGoals
    public function getGoals(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        if (!empty($request->id)) {
            // Check if any campaigns exist for the given id
            $campaigns = Goals::where('id', $request->id)
                ->get();

            // If campaigns exist, process the data
            if ($campaigns->isNotEmpty()) {
                foreach ($campaigns as $campaign) {
                    $campaign->start_date = $campaign->start_date ?? "";
                    $campaign->end_date = $campaign->end_date ?? "";
                    $campaign->days = $campaign->days ?? "";
                    $campaign->price = $campaign->price ?? "";
                }

                return response()->json([
                    'status' => true,
                    'message' => 'Goals Found',
                    'GoalData' => $campaigns,
                ]);
            } else {
                return response()->json([
                    'status' => false,
                    'message' => 'No Goals found ',
                    'GoalData' => [],
                ]);
            }
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Please provide a id',
                'GoalData' => [],
            ]);
        }
    }




    // getPaymentConfig
    // getPaymentConfig
    // public function getPaymentConfig()
    // {
    //     // Fetch all records from the payment_gateway_key table
    //     $paymentConfigs = PaymentGatewayKey::all();

    //     if ($paymentConfigs->isNotEmpty()) {
    //         // Ensure all fields have default values if null
    //         $paymentConfigs->transform(function ($config) {
    //             $config->text = $config->text ?? "";
    //             $config->public_key = $config->public_key ?? "";
    //             $config->secret_key = $config->secret_key ?? "";
    //             $config->mode = $config->mode ?? "";
    //             $config->status = $config->status ?? "";
    //             return $config;
    //         });

    //         return response()->json([
    //             'status' => true,
    //             'message' => 'Payment Configurations Found',
    //             'data' => $paymentConfigs,
    //         ]);
    //     } else {
    //         return response()->json([
    //             'status' => false,
    //             'message' => 'No Payment Configurations found',
    //             'data' => [],
    //         ]);
    //     }
    // }


    public function getPaymentConfig(Request $request)
    {
        $categories = PaymentGatewayKey::get(); // Fetch all records

        if ($categories->isEmpty()) {
            // return $this->sendResponse((object)[], "No Payment Gateway Keys Found");

            return response()->json([
                'success' => "false",
                'message' => 'No Payment Gateway Keys Found',
                'data' => [],
            ]);
        }

        $result = [];

        foreach ($categories as $category) {
            $result[$category->text] = [ // Using text as key, change to id if needed
                "public_key" => (string)$category->public_key,
                "secret_key" => (string)$category->secret_key,
                "mode" => (string)$category->mode,
                "country_code" => (string)($category->country_code ?? ""),
                "currency_code" => (string)($category->currency_code ?? ""),
                "status" => (int)($category->status ?? 0),
            ];
        }

        // return $this->sendResponse((object)$result, "Payment Gateway Keys Retrieved");


        return response()->json([
            'success' => "true",
            'message' => 'Payment Gateway Keys Retrieved',
            'data' => $result,
        ]);
    }





    // public function stripeIntent(Request $request)
    // {
    //     // Get user_id from request
    //     $myId = $request->input('user_id');
    //     if (!$myId) {
    //         return $this->sendError("User ID Missing", "User ID is required", 400);
    //     }

    //     // Get total amount from request
    //     $total = $request->input('total', 0);
    //     if ($total <= 0) {
    //         return $this->sendError("Invalid Amount", "Total amount must be greater than zero", 400);
    //     }

    //     \Stripe\ApiRequestor::setHttpClient(new \Stripe\HttpClient\CurlClient([CURLOPT_SSL_VERIFYPEER => false]));


    //     try {
    //         // ✅ Set API Key before initializing Stripe
    //         \Stripe\Stripe::setApiKey('sk_test_51OP303SJayPbST1lMDr6nn6WieehdLmIpiG2pgVil38DVNPjDqKcFG87d1GMOk10WWtoqZIxvSx2WLAP7G1GMkWu00SJWzq7cn');

    //         // ✅ Disable SSL verification for cURL
    //         $ch = curl_init();
    //         curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    //         curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);

    //         // Create payment intent
    //         $stripe = new \Stripe\StripeClient('sk_test_51OP303SJayPbST1lMDr6nn6WieehdLmIpiG2pgVil38DVNPjDqKcFG87d1GMOk10WWtoqZIxvSx2WLAP7G1GMkWu00SJWzq7cn');
    //         $res = $stripe->paymentIntents->create([
    //             'amount' => (int)$total * 100,
    //             'currency' => "usd",
    //             'description' => "Nylitical Subscription",
    //             'automatic_payment_methods' => ['enabled' => true],
    //             'shipping' => [
    //                 'name' => "Random Singh",
    //                 'address' => [
    //                     'line1' => "510 Townsend St",
    //                     'postal_code' => "98140",
    //                     'city' => "San Francisco",
    //                     'state' => "CA",
    //                     'country' => "US",
    //                 ],
    //             ],
    //         ]);

    //         return $this->sendresponse(['clientSecret' => $res->client_secret], 'Response');
    //     } catch (\Throwable $th) {
    //         return $this->sendError("Payment Error", $th->getMessage());
    //     }
    // }


    public function stripeIntent(Request $request)
    {
        // Get user_id from request
        // $myId = $request->input('user_id');
        $user = Auth::user();
        $myId = $user->id;
        // if (!$myId) {
        //     return $this->sendError("User ID Missing", "User ID is required", 400);
        // }

        // Get total amount from request
        $total = $request->input('total', 0);
        if ($total <= 0) {
            return $this->sendError("Invalid Amount", "Total amount must be greater than zero", 400);
        }

        // Fetch user email
        $user = \App\Models\User::find($myId);
        if (!$user) {
            return $this->sendError("User Not Found", "No user found with the given ID", 404);
        }

        $email = $user->email;
        $firstName = $user->first_name ?? 'Customer';
        $lastName = $user->last_name ?? '';

        \Stripe\ApiRequestor::setHttpClient(new \Stripe\HttpClient\CurlClient([CURLOPT_SSL_VERIFYPEER => false]));

        try {
            \Stripe\Stripe::setApiKey('sk_test_51OP303SJayPbST1lMDr6nn6WieehdLmIpiG2pgVil38DVNPjDqKcFG87d1GMOk10WWtoqZIxvSx2WLAP7G1GMkWu00SJWzq7cn');

            // Create Stripe client
            $stripe = new \Stripe\StripeClient('sk_test_51OP303SJayPbST1lMDr6nn6WieehdLmIpiG2pgVil38DVNPjDqKcFG87d1GMOk10WWtoqZIxvSx2WLAP7G1GMkWu00SJWzq7cn');

            // ✅ Create Stripe customer
            $customer = $stripe->customers->create([
                'email' => $email,
                'name' => $firstName . ' ' . $lastName,
            ]);

            // ✅ Create payment intent linked to that customer
            $res = $stripe->paymentIntents->create([
                'amount' => (int)$total * 100,
                'currency' => "usd",
                'description' => "Nylitical Subscription",
                'customer' => $customer->id,
                'receipt_email' => $email,
                'automatic_payment_methods' => ['enabled' => true],
                'shipping' => [
                    'name' => $firstName,
                    'address' => [
                        'line1' => "510 Townsend St",
                        'postal_code' => "98140",
                        'city' => "San Francisco",
                        'state' => "CA",
                        'country' => "US",
                    ],
                ],
            ]);

            return $this->sendresponse(['clientSecret' => $res->client_secret], 'Response');
        } catch (\Throwable $th) {
            return $this->sendError("Payment Error", $th->getMessage());
        }
    }





    // public function stripeIntent(Request $request)
    // {
    //     try {
    //         Stripe::setApiKey('sk_test_51OP303SJayPbST1lMDr6nn6WieehdLmIpiG2pgVil38DVNPjDqKcFG87d1GMOk10WWtoqZIxvSx2WLAP7G1GMkWu00SJWzq7cn');

    //         \Stripe\ApiRequestor::setHttpClient(new \Stripe\HttpClient\CurlClient([CURLOPT_SSL_VERIFYPEER => false]));
    //         $paymentIntent = PaymentIntent::create([
    //             'amount' => 1000, // Amount in cents (e.g., $10.00)
    //             'currency' => 'usd',
    //             'payment_method_types' => ['card'],
    //         ]);

    //         return response()->json([
    //             'clientSecret' => $paymentIntent->client_secret,
    //         ]);
    //     } catch (\Exception $e) {
    //         return response()->json(['error' => $e->getMessage()], 500);
    //     }
    // }




    // addNewsEmail
    // addNewsEmail
    public function addNewsEmail(Request $request)
    {
        $rules = [
            'email' => 'required|email|unique:news_email,email',
        ];

        $customMessages = [
            'email.required' => 'Please enter the email.',
            'email.email' => 'Please enter a valid email address.',
            'email.unique' => 'This email is already subscribed.',
        ];

        $validator = Validator::make($request->all(), $rules, $customMessages);

        if ($validator->fails()) {
            return response()->json(
                [
                    'status' => false,
                    'message' => $validator->errors()->first(),
                ],
                400
            );
        }

        // Save email in the database
        $data = new NewsEmail();
        $data->email = $request->email;
        $data->save();

        return response()->json(
            [
                'status' => true,
                'message' => 'Your email has been successfully subscribed!',
            ],
            200
        );
    }




    public function showPriceRange(Request $request)
    {
        $minPrice = Stores::whereNotNull('price')->pluck('price')->map(function ($price) {
            return (float) $price;
        })->min();

        $maxPrice = Stores::whereNotNull('price')->pluck('price')->map(function ($price) {
            return (float) $price;
        })->max();

        $defaultCurrency = SiteSetup::first()->default_currency ?? '$';

        $formattedMinPrice = $defaultCurrency . number_format($minPrice, 0, '', ',');
        $formattedMaxPrice = $defaultCurrency . number_format($maxPrice, 0, '', ',');

        return response()->json([
            'status' => true,
            'min_price' => $formattedMinPrice,
            'max_price' => $formattedMaxPrice,
        ], 200);
    }





    // getGoalsPayment
    public function getGoalsPayment(Request $request)
    {

        $user = Auth::user();
        $vendor_id = $user->id;
        if (!empty($vendor_id)) {
            // Fetch goal_id from GoalPayment table using vendor_id
            $goalIds = GoalPaymentHistory::where('vendor_id', $vendor_id)
                ->pluck('goal_id');

            // Fetch goals data using goal_id
            $goals = Goals::whereIn('id', $goalIds)->get();

            if ($goals->isNotEmpty()) {
                foreach ($goals as $goal) {
                    $goal->start_date = $goal->start_date ?? "";
                    $goal->end_date = $goal->end_date ?? "";
                    $goal->days = $goal->days ?? "";
                    $goal->price = $goal->price ?? "";
                }

                return response()->json([
                    'status' => true,
                    'message' => 'Goals Found',
                    'GoalData' => $goals,
                ]);
            } else {
                return response()->json([
                    'status' => false,
                    'message' => 'No Goals found',
                    'GoalData' => [],
                ]);
            }
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Please provide a vendor_id',
                'GoalData' => [],
            ]);
        }
    }




    // getSubscribePayment
    public function getSubscribePayment(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        if (!empty($user_id)) {
            // $user_id = $request->user_id;

            // Fetch user subscription data based on user_id
            // $subscriptions = UserSubscription::where('user_id', $user_id)->get();
            $subscriptions = UserSubscriptionHistory::where('user_id', $user_id)->orderByDesc('id')->get();

            if ($subscriptions->isNotEmpty()) {
                foreach ($subscriptions as $subscription) {
                    $subscription->start_date = $subscription->start_date ?? "";
                    $subscription->expire_date = $subscription->expire_date ?? "";
                    $subscription->plan_name = $subscription->plan_name ?? "";
                    $subscription->price = $subscription->price ?? "";
                    $subscription->payment_mode = $subscription->payment_mode ?? "";
                }

                return response()->json([
                    'status' => true,
                    'message' => 'Payment subscription found',
                    'data' => $subscriptions,
                ]);
            } else {
                return response()->json([
                    'status' => false,
                    'message' => 'No payment history found',
                    'data' => [],
                ]);
            }
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Please provide a valid user_id',
                'data' => [],
            ]);
        }
    }


    public function getallSubscribePayment(Request $request)
    {
        // if (!empty($request->user_id)) {
        $user = Auth::user();
        $user_id = $user->id;
        $service_id = $request->service_id;

        $vendor = Service::where('id', $service_id)->first();
        $user_id = $vendor->vendor_id;

        // Fetch user subscription data based on user_id
        $subscriptions = UserSubscriptionHistory::where('user_id', $user_id)->get();

        if ($subscriptions->isNotEmpty()) {
            foreach ($subscriptions as $subscription) {
                $subscription->start_date = $subscription->start_date ?? "";
                $subscription->expire_date = $subscription->expire_date ?? "";
                $subscription->plan_name = $subscription->plan_name ?? "";
                $subscription->price = $subscription->price ?? "";
                $subscription->payment_mode = $subscription->payment_mode ?? "";
                $subscription->status = $subscription->status ?? "";

                $time_stamp = now()->format('Y-m-d');

                $done = UserSubscriptionHistory::where($subscription->end_date_do, '>=', $time_stamp)->first();
                $subscription->all_status = $done ? "1" : "0";
            }

            return response()->json([
                'status' => true,
                'message' => 'Subcription List found',
                'data' => $subscriptions,
            ]);
        } else {
            return response()->json([
                'status' => false,
                'message' => 'No Subcription history found',
                'data' => [],
            ]);
        }
    }


    public function getMetaTitleDesc(Request $request)
    {
        $categories = HomeSeo::all(); // Fetch all records

        if ($categories->isEmpty()) {
            return $this->sendResponse((object)[], "No SEO Data Found");
        }

        $result = [];

        foreach ($categories as $category) {
            $result[$category->id] = [ // Using id as key
                "title" => $category->title,
                "body" => $category->body,
                "image" => asset('assets/images/index/' . $category->image), // Fix concatenation and typo
            ];
        }

        return $this->sendResponse((object)$result, "SEO Data Retrieved Successfully");
    }


    // public function createStripeSubscription2(Request $request)
    // {
    //     $userId = $request->user_id;
    //     $user = User::find($userId);
    //     $priceId = $request->price;

    //     if (!$user) {
    //         return $this->sendError("User not found.");
    //     }

    //     \Stripe\Stripe::setApiKey('sk_test_51OP303SJayPbST1lMDr6nn6WieehdLmIpiG2pgVil38DVNPjDqKcFG87d1GMOk10WWtoqZIxvSx2WLAP7G1GMkWu00SJWzq7cn');
    //     $stripe = new \Stripe\StripeClient('sk_test_51OP303SJayPbST1lMDr6nn6WieehdLmIpiG2pgVil38DVNPjDqKcFG87d1GMOk10WWtoqZIxvSx2WLAP7G1GMkWu00SJWzq7cn');

    //     if (!$user->stripe_customer_id) {
    //         $customer = $stripe->customers->create([
    //             'email' => $user->email,
    //             'name' => $user->name,
    //         ]);
    //         $user->stripe_customer_id = $customer->id;
    //         $user->save();
    //     }

    //     $subscription = $stripe->subscriptions->create([
    //         'customer' => $user->stripe_customer_id,
    //         'items' => [[
    //             'price' => $priceId,
    //         ]],
    //         'payment_behavior' => 'default_incomplete', // ✅ corrected
    //         'default_payment_method' => 'pm_card_visa', // test card
    //         'expand' => ['latest_invoice.payment_intent'],
    //     ]);

    //     return response()->json([
    //         'subscriptionId' => $subscription->id,
    //         'clientSecret' => $subscription->latest_invoice->payment_intent->client_secret,
    //     ]);
    // }


    // public function createStripeSubscription25(Request $request)
    // {
    //     $userId = $request->user_id;
    //     $priceId = $request->price;
    //     $paymentMethodId = '2'; // ✅ frontend-created PM ID

    //     $user = User::find($userId);
    //     if (!$user) {
    //         return response()->json(['error' => 'User not found.'], 404);
    //     }

    //     \Stripe\Stripe::setApiKey('sk_test_51OP303SJayPbST1lMDr6nn6WieehdLmIpiG2pgVil38DVNPjDqKcFG87d1GMOk10WWtoqZIxvSx2WLAP7G1GMkWu00SJWzq7cn');
    //     $stripe = new \Stripe\StripeClient('sk_test_51OP303SJayPbST1lMDr6nn6WieehdLmIpiG2pgVil38DVNPjDqKcFG87d1GMOk10WWtoqZIxvSx2WLAP7G1GMkWu00SJWzq7cn');

    //     // Create Stripe Customer if not exists
    //     if (!$user->stripe_customer_id) {
    //         $customer = $stripe->customers->create([
    //             'email' => $user->email,
    //             'name'  => $user->name,
    //         ]);
    //         $user->stripe_customer_id = $customer->id;
    //         $user->save();
    //     }

    //     // Attach the payment method to customer
    //     $stripe->paymentMethods->attach($paymentMethodId, [
    //         'customer' => $user->stripe_customer_id,
    //     ]);

    //     // Set default payment method
    //     $stripe->customers->update($user->stripe_customer_id, [
    //         'invoice_settings' => [
    //             'default_payment_method' => $paymentMethodId,
    //         ],
    //     ]);

    //     // Create subscription
    //     $subscription = $stripe->subscriptions->create([
    //         'customer' => $user->stripe_customer_id,
    //         'items' => [[
    //             'price' => $priceId,
    //         ]],
    //         'default_payment_method' => $paymentMethodId,
    //         'payment_behavior' => 'default_incomplete',
    //         'expand' => ['latest_invoice.payment_intent'],
    //     ]);

    //     return response()->json([
    //         'subscriptionId' => $subscription->id,
    //         'clientSecret' => $subscription->latest_invoice->payment_intent->client_secret,
    //     ]);
    // }

    // public function createStripeSubscriptionWorking(Request $request)
    // {
    //     $userId = $request->user_id;
    //     $priceId = $request->price;

    //     $user = User::find($userId);
    //     if (!$user) {
    //         return response()->json(['error' => 'User not found.'], 404);
    //     }

    //     \Stripe\Stripe::setApiKey('sk_test_51OP303SJayPbST1lMDr6nn6WieehdLmIpiG2pgVil38DVNPjDqKcFG87d1GMOk10WWtoqZIxvSx2WLAP7G1GMkWu00SJWzq7cn');
    //     $stripe = new \Stripe\StripeClient('sk_test_51OP303SJayPbST1lMDr6nn6WieehdLmIpiG2pgVil38DVNPjDqKcFG87d1GMOk10WWtoqZIxvSx2WLAP7G1GMkWu00SJWzq7cn');

    //     // Create customer if not exists
    //     if (!$user->stripe_customer_id) {
    //         $customer = $stripe->customers->create([
    //             'email' => $user->email,
    //             'name'  => $user->name,
    //         ]);
    //         $user->stripe_customer_id = $customer->id;
    //         $user->save();
    //     }

    //     // Create subscription WITHOUT default payment method
    //     $subscription = $stripe->subscriptions->create([
    //         'customer' => $user->stripe_customer_id,
    //         'items' => [[
    //             'price' => $priceId,
    //         ]],
    //         'payment_behavior' => 'default_incomplete', // won't charge now
    //         'expand' => ['latest_invoice.payment_intent'],
    //     ]);

    //     return response()->json([
    //         'subscriptionId' => $subscription->id,
    //         'clientSecret' => $subscription->latest_invoice->payment_intent->client_secret ?? null,
    //         'message' => 'Subscription created. Awaiting payment method.',
    //     ]);
    // }



    // public function createStripeSubscription(Request $request)
    // {
    //     $userId = $request->user_id;
    //     $priceId = $request->price;
    //     $paymentMethodId = $request->payment_method; // ✅ from frontend

    //     $user = User::find($userId);

    //     if (!$user) {
    //         return response()->json(['error' => 'User not found.'], 404);
    //     }

    //     \Stripe\Stripe::setApiKey('sk_test_51OP303SJayPbST1lMDr6nn6WieehdLmIpiG2pgVil38DVNPjDqKcFG87d1GMOk10WWtoqZIxvSx2WLAP7G1GMkWu00SJWzq7cn');
    //     $stripe = new \Stripe\StripeClient('sk_test_51OP303SJayPbST1lMDr6nn6WieehdLmIpiG2pgVil38DVNPjDqKcFG87d1GMOk10WWtoqZIxvSx2WLAP7G1GMkWu00SJWzq7cn');

    //     // Step 1: Create customer if not exists
    //     if (!$user->stripe_customer_id) {
    //         $customer = $stripe->customers->create([
    //             'email' => $user->email,
    //             'name' => $user->name,
    //         ]);
    //         $user->stripe_customer_id = $customer->id;
    //         $user->save();
    //     }

    //     // Step 2: Attach the payment method (only if it's passed and not already attached)
    //     try {
    //         $stripe->paymentMethods->attach($paymentMethodId, [
    //             'customer' => $user->stripe_customer_id
    //         ]);
    //     } catch (\Exception $e) {
    //         // If already attached, this might throw an error; you can log or ignore it
    //     }

    //     // Step 3: Set it as default
    //     $stripe->customers->update($user->stripe_customer_id, [
    //         'invoice_settings' => [
    //             'default_payment_method' => $paymentMethodId,
    //         ]
    //     ]);

    //     // Step 4: Create subscription
    //     $subscription = $stripe->subscriptions->create([
    //         'customer' => $user->stripe_customer_id,
    //         'items' => [[
    //             'price' => $priceId,
    //         ]],
    //         'default_payment_method' => $paymentMethodId,
    //         'expand' => ['latest_invoice.payment_intent'],
    //     ]);

    //     return response()->json([
    //         'subscriptionId' => $subscription->id,
    //         'clientSecret' => $subscription->latest_invoice->payment_intent->client_secret ?? null,
    //     ]);
    // }


    // public function createStripeSubscription22(Request $request)
    // {
    //     $userId = $request->user_id;
    //     $priceId = $request->price;
    //     $paymentMethodId = $request->payment_method; // ✅ from frontend

    //     $user = User::find($userId);

    //     if (!$user) {
    //         return response()->json(['error' => 'User not found.'], 404);
    //     }

    //     \Stripe\Stripe::setApiKey('sk_test_51OP303SJayPbST1lMDr6nn6WieehdLmIpiG2pgVil38DVNPjDqKcFG87d1GMOk10WWtoqZIxvSx2WLAP7G1GMkWu00SJWzq7cn');
    //     $stripe = new \Stripe\StripeClient('sk_test_51OP303SJayPbST1lMDr6nn6WieehdLmIpiG2pgVil38DVNPjDqKcFG87d1GMOk10WWtoqZIxvSx2WLAP7G1GMkWu00SJWzq7cn');

    //     // Step 1: Create customer if not exists
    //     if (!$user->stripe_customer_id) {
    //         $customer = $stripe->customers->create([
    //             'email' => $user->email,
    //             'name' => $user->name,
    //         ]);
    //         $user->stripe_customer_id = $customer->id;
    //         $user->save();
    //     }

    //     // Step 2: Attach the payment method (only if it's passed and not already attached)
    //     try {
    //         $stripe->paymentMethods->attach($paymentMethodId, [
    //             'customer' => $user->stripe_customer_id
    //         ]);
    //     } catch (\Exception $e) {
    //         // Already attached or failed
    //     }

    //     // Step 3: Set it as default payment method
    //     $stripe->customers->update($user->stripe_customer_id, [
    //         'invoice_settings' => [
    //             'default_payment_method' => $paymentMethodId,
    //         ]
    //     ]);

    //     // Step 4: Create the subscription
    //     $subscription = $stripe->subscriptions->create([
    //         'customer' => $user->stripe_customer_id,
    //         'items' => [[
    //             'price' => $priceId,
    //         ]],
    //         'default_payment_method' => $paymentMethodId,
    //         'expand' => ['latest_invoice.payment_intent'],
    //     ]);

    //     $paymentIntent = $subscription->latest_invoice->payment_intent ?? null;

    //     // Step 5: Handle payment result
    //     if ($paymentIntent) {
    //         if ($paymentIntent->status === 'requires_action' && $paymentIntent->next_action->type === 'use_stripe_sdk') {
    //             return response()->json([
    //                 'requires_action' => true,
    //                 'clientSecret' => $paymentIntent->client_secret,
    //                 'message' => 'Authentication (3D Secure) required',
    //             ]);
    //         }

    //         if ($paymentIntent->status !== 'succeeded') {
    //             return response()->json([
    //                 'error' => 'Initial payment failed. Please try another payment method.',
    //                 'status' => $paymentIntent->status,
    //             ], 400);
    //         }
    //     }

    //     return response()->json([
    //         'subscriptionId' => $subscription->id,
    //         'status' => $subscription->status, // should be 'active'
    //         'clientSecret' => $paymentIntent->client_secret ?? null,
    //     ]);
    // }


    // public function createStripeSubscription(Request $request)
    // {
    //     $priceId = $request->price;
    //     $paymentMethodId = $request->payment_method;
    //     $email = $request->email;

    //     if (!$email) {
    //         return response()->json(['error' => 'Email is required.'], 400);
    //     }

    //     \Stripe\Stripe::setApiKey('sk_test_51OP303SJayPbST1lMDr6nn6WieehdLmIpiG2pgVil38DVNPjDqKcFG87d1GMOk10WWtoqZIxvSx2WLAP7G1GMkWu00SJWzq7cn');
    //     $stripe = new \Stripe\StripeClient('sk_test_51OP303SJayPbST1lMDr6nn6WieehdLmIpiG2pgVil38DVNPjDqKcFG87d1GMOk10WWtoqZIxvSx2WLAP7G1GMkWu00SJWzq7cn');

    //     // Step 1: Try to find customer by email
    //     try {
    //         $customers = $stripe->customers->all(['limit' => 100]);
    //         $matchingCustomer = collect($customers->data)->firstWhere('email', $email);
    //     } catch (\Exception $e) {
    //         return response()->json(['error' => 'Failed to search for Stripe customer.', 'details' => $e->getMessage()], 400);
    //     }

    //     // Step 2: Create customer if not found
    //     if (!$matchingCustomer) {
    //         try {
    //             $matchingCustomer = $stripe->customers->create([
    //                 'email' => $email,
    //             ]);
    //         } catch (\Exception $e) {
    //             return response()->json(['error' => 'Failed to create Stripe customer.', 'details' => $e->getMessage()], 400);
    //         }
    //     }

    //     $stripeCustomerId = $matchingCustomer->id;

    //     // Step 3: Attach the payment method
    //     try {
    //         $stripe->paymentMethods->attach($paymentMethodId, [
    //             'customer' => $stripeCustomerId
    //         ]);
    //     } catch (\Exception $e) {
    //         // Ignore if already attached
    //     }

    //     // Step 4: Set default payment method
    //     try {
    //         $stripe->customers->update($stripeCustomerId, [
    //             'invoice_settings' => [
    //                 'default_payment_method' => $paymentMethodId,
    //             ]
    //         ]);
    //     } catch (\Exception $e) {
    //         return response()->json(['error' => 'Failed to set default payment method.', 'details' => $e->getMessage()], 400);
    //     }

    //     // Step 5: Create the subscription
    //     try {
    //         $subscription = $stripe->subscriptions->create([
    //             'customer' => $stripeCustomerId,
    //             'items' => [[
    //                 'price' => $priceId,
    //             ]],
    //             'default_payment_method' => $paymentMethodId,
    //             'expand' => ['latest_invoice.payment_intent'],
    //         ]);
    //     } catch (\Exception $e) {
    //         return response()->json(['error' => 'Subscription creation failed.', 'details' => $e->getMessage()], 400);
    //     }

    //     $paymentIntent = $subscription->latest_invoice->payment_intent ?? null;

    //     // Step 6: Handle payment result
    //     if ($paymentIntent) {
    //         if ($paymentIntent->status === 'requires_action' && $paymentIntent->next_action->type === 'use_stripe_sdk') {
    //             return response()->json([
    //                 'requires_action' => true,
    //                 'clientSecret' => $paymentIntent->client_secret,
    //                 'message' => 'Authentication (3D Secure) required',
    //             ]);
    //         }

    //         if ($paymentIntent->status !== 'succeeded') {
    //             return response()->json([
    //                 'error' => 'Initial payment failed. Please try another payment method.',
    //                 'status' => $paymentIntent->status,
    //             ], 400);
    //         }
    //     }

    //     return response()->json([
    //         'subscriptionId' => $subscription->id,
    //         'status' => $subscription->status,
    //         'clientSecret' => $paymentIntent->client_secret ?? null,
    //         'customerId' => $stripeCustomerId,
    //     ]);
    // }






    // public function createStripeSubscription(Request $request)
    // {
    //     $userId = $request->user_id;
    //     $priceId = $request->price;

    //     $user = User::find($userId);

    //     if (!$user) {
    //         return response()->json(['error' => 'User not found.'], 404);
    //     }

    //     \Stripe\Stripe::setApiKey('sk_test_51OP303SJayPbST1lMDr6nn6WieehdLmIpiG2pgVil38DVNPjDqKcFG87d1GMOk10WWtoqZIxvSx2WLAP7G1GMkWu00SJWzq7cn');
    //     $stripe = new \Stripe\StripeClient('sk_test_51OP303SJayPbST1lMDr6nn6WieehdLmIpiG2pgVil38DVNPjDqKcFG87d1GMOk10WWtoqZIxvSx2WLAP7G1GMkWu00SJWzq7cn');

    //     // Step 1: Create customer if not exists
    //     if (!$user->stripe_customer_id) {
    //         $customer = $stripe->customers->create([
    //             'email' => $user->email,
    //             'name' => $user->name,
    //         ]);
    //         $user->stripe_customer_id = $customer->id;
    //         $user->save();
    //     }

    //     // Step 2: Create subscription without setting payment method explicitly
    //     try {
    //         $subscription = $stripe->subscriptions->create([
    //             'customer' => $user->stripe_customer_id,
    //             'items' => [[
    //                 'price' => $priceId,
    //             ]],
    //             'expand' => ['latest_invoice.payment_intent'],
    //         ]);
    //     } catch (\Exception $e) {
    //         return response()->json(['error' => $e->getMessage()], 400);
    //     }

    //     $paymentIntent = $subscription->latest_invoice->payment_intent ?? null;

    //     if ($paymentIntent) {
    //         if ($paymentIntent->status === 'requires_action' && $paymentIntent->next_action->type === 'use_stripe_sdk') {
    //             return response()->json([
    //                 'requires_action' => true,
    //                 'clientSecret' => $paymentIntent->client_secret,
    //                 'message' => 'Authentication (3D Secure) required',
    //             ]);
    //         }

    //         if ($paymentIntent->status !== 'succeeded') {
    //             return response()->json([
    //                 'error' => 'Initial payment failed. Please try another payment method.',
    //                 'status' => $paymentIntent->status,
    //             ], 400);
    //         }
    //     }

    //     return response()->json([
    //         'subscriptionId' => $subscription->id,
    //         'status' => $subscription->status,
    //         'clientSecret' => $paymentIntent->client_secret ?? null,
    //     ]);
    // }

    public function removeUserImages(Request $request)
    {
        // Get the authenticated vendor's ID
        // $user_id = $request->user_id;
        $user = Auth::user();
        $user_id = $user->id;

        // Check if the authenticated vendor ID is available
        if ($user_id) {
            // Validate the request to ensure product_id and product_image_id are provided
            // $request->validate([
            //     'user_id' => 'required|exists:users,id',
            // ]);

            // Check if the product exists and belongs to the authenticated vendor
            $productExists = User::where('id', $user_id)
                ->exists();

            if ($productExists) {
                // Verify that the image belongs to the product and vendor by checking product_images table
                $userImage = User::where('id', $user_id)
                    ->first();

                // Check if the product image exists
                if ($userImage) {
                    // Remove the product image
                    // Empty the image field only (keep the record)
                    $userImage->image = '';
                    $userImage->save();

                    return response()->json([
                        'success' => true,
                        'message' => 'User image removed successfully!',
                    ], 200);
                } else {
                    return response()->json([
                        'success' => false,
                        'message' => 'User image not found or it does not belong to this product.',
                    ], 404);
                }
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found or you do not have permission to delete images for this product.',
                ], 404);
            }
        } else {
            return response()->json([
                'success' => false,
                'message' => 'User not authenticated.',
            ], 401);
        }
    }
}
