<?php

namespace App\Http\Controllers\Api;

use Illuminate\Support\Str;
use Illuminate\Support\Facades\Validator;
use App\Models\User;
use App\Http\Controllers\Controller;
use App\Models\About;
use App\Models\AppFeedback;
use App\Models\CancelPolicy;
use App\Models\Category;
use App\Models\ClientReviews;
use App\Models\ContactUs;
use App\Models\Currencies;
use App\Models\CustomerSupport;
use App\Models\Faq;
use App\Models\GoalPayment;
use App\Models\IndexCards;
use App\Models\IndexCategories;
use App\Models\IndexCities;
use Illuminate\Support\Facades\Hash;
use App\Models\IndexSlider;
use App\Models\MailSetup;
use App\Models\MobileUrl;
use App\Models\NewsEmail;
use App\Models\PaymentGatewayKey;
use App\Models\RefundPolicy;
use App\Models\ReportText;
use App\Models\Service;
use App\Models\Admin;
use App\Models\ServiceImages;
use App\Models\ServiceLike;
use App\Models\ServiceReview;
use App\Models\SiteSetup;
use App\Models\Slides;
use App\Models\SocialMedia;
use App\Models\Stores;
use App\Models\SubCategory;
use App\Models\SubscriptionPlan;
use App\Models\UserSubscription;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Stripe\Subscription;
use Illuminate\Support\Facades\DB;
use App\Models\StoreImages;
use App\Models\ServiceLead;
use App\Models\PrivacyPolicy;
use App\Models\TermsCondition;
use App\Models\NearbyDistance;
use App\Models\IndexSponserStores;
use App\Models\IndexBanner;
use App\Models\IndexPerfectStore;
use App\Models\IndexTestimonials;
use App\Models\HomeSeo;
use App\Models\HomeSettings;
use App\Models\Goals;
use App\Models\Campaign;
use App\Models\UserSubscriptionHistory;
use App\Models\GoalPaymentHistory;
use App\Models\UserLoginStatus;
use PHPOpenSourceSaver\JWTAuth\Facades\JWTAuth;
use Illuminate\Support\Facades\Auth;


class AdminApiController extends Controller
{
    // customerList
    public function customerList(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $search = $request->input('search'); // e.g., 'A'
        $page = $request->input('page', 1); // Default page = 1
        $pageSize = $request->input('pageSize', 10); // Default pageSize = 10
        $orderBy = strtoupper($request->input('orderBy', 'DESC')); // Default sort = DESC

        // Ensure only valid order values
        if (!in_array($orderBy, ['ASC', 'DESC'])) {
            $orderBy = 'DESC';
        }

        $query = User::select('id', 'first_name', 'last_name', 'username', 'created_at', 'mobile', 'image', 'status', 'email')
            ->where('role', 'user');

        // Filter by first_name's first letter or email if 'search' is provided
        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('first_name', 'LIKE', $search . '%')
                    ->orWhere('email', 'LIKE', '%' . $search . '%')
                    ->orWhereRaw("CONCAT(first_name, ' ', last_name) LIKE ?", ["%$search%"]);
            });
        }

        $total = $query->count();

        $customers = $query->orderBy('created_at', $orderBy)
            ->skip(($page - 1) * $pageSize)
            ->take($pageSize)
            ->get()
            ->map(function ($user) {
                return [
                    'id' => $user->id,
                    'first_name' => $user->first_name ?: '',
                    'last_name' => $user->last_name ?: '',
                    'username' => $user->username ?: '',
                    'email' => $user->email ?: '',
                    'created_at' => $user->created_at ?: '',
                    'mobile' => $user->mobile ?: '',
                    'image' => $user->image
                        ? url('assets/images/users_images/' . $user->image)
                        : "",
                    'status' => $user->status,
                ];
            });

        return response()->json([
            'status' => true,
            'message' => 'Customer list fetched successfully',
            'customers' => $customers,
            'Pagination' => [
                'total_pages' => ceil($total / $pageSize),
                'total_records' => $total,
                'current_page' => (int) $page,
                'records_per_page' => (int) $pageSize,

            ],
        ]);
    }



    // vendorList
    public function vendorList(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        $search = $request->input('search'); // e.g., 'A'
        $page = $request->input('page', 1); // Default page = 1
        $pageSize = $request->input('pageSize', 10); // Default pageSize = 10
        $orderBy = strtoupper($request->input('orderBy', 'DESC')); // Default sort = DESC

        // Ensure only valid order values
        if (!in_array($orderBy, ['ASC', 'DESC'])) {
            $orderBy = 'DESC';
        }

        $query = User::select('id', 'first_name', 'last_name', 'username', 'created_at', 'mobile', 'image', 'status', 'email')
            ->where('role', 'vendor');

        // Filter by first_name's first letter or email if 'search' is provided
        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('first_name', 'LIKE', $search . '%')
                    ->orWhere('email', 'LIKE', '%' . $search . '%')
                    ->orWhereRaw("CONCAT(first_name, ' ', last_name) LIKE ?", ["%$search%"]);
            });
        }

        $total = $query->count();

        $customers = $query->orderBy('created_at', $orderBy)
            ->skip(($page - 1) * $pageSize)
            ->take($pageSize)
            ->get()
            ->map(function ($user) {
                return [
                    'id' => $user->id,
                    'first_name' => $user->first_name ?: '',
                    'last_name' => $user->last_name ?: '',
                    'username' => $user->username ?: '',
                    'email' => $user->email ?: '',
                    'created_at' => $user->created_at ?: '',
                    'mobile' => $user->mobile ?: '',
                    'image' => $user->image
                        ? url('assets/images/users_images/' . $user->image)
                        : "",
                    'status' => $user->status,
                ];
            });

        return response()->json([
            'status' => true,
            'message' => 'Vendor list fetched successfully',
            'vendors' => $customers,
            'Pagination' => [
                'total_pages' => ceil($total / $pageSize),
                'total_records' => $total,
                'current_page' => (int) $page,
                'records_per_page' => (int) $pageSize,
            ],
        ]);
    }


    public function vendorListWithoutPagination(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $search = $request->input('search'); // e.g., 'A'
        $orderBy = strtoupper($request->input('orderBy', 'DESC')); // Default sort = DESC

        // Ensure only valid order values
        if (!in_array($orderBy, ['ASC', 'DESC'])) {
            $orderBy = 'DESC';
        }

        $query = User::select('id', 'first_name', 'last_name', 'username', 'created_at', 'mobile', 'image', 'status', 'email')
            ->where('role', 'vendor')
            ->orwhere('role', 'admin');

        // Filter by first_name's first letter or email if 'search' is provided
        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('first_name', 'LIKE', $search . '%')
                    ->orWhere('email', 'LIKE', '%' . $search . '%');
            });
        }

        $vendors = $query->orderBy('created_at', $orderBy)
            ->get()
            ->map(function ($user) {
                return [
                    'id' => $user->id,
                    'first_name' => $user->first_name ?: '',
                    'last_name' => $user->last_name ?: '',
                    'username' => $user->username ?: '',
                    'email' => $user->email ?: '',
                    'created_at' => $user->created_at ?: '',
                    'mobile' => $user->mobile ?: '',
                    'image' => $user->image
                        ? url('assets/images/users_images/' . $user->image)
                        : "",
                    'status' => $user->status,
                ];
            });

        return response()->json([
            'status' => true,
            'message' => 'Vendor list fetched successfully',
            'vendors' => $vendors,
        ]);
    }



    // allUserList
    public function allUserList(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $search = $request->input('search'); // e.g., 'A'
        $page = $request->input('page', 1); // Default page = 1
        $pageSize = $request->input('pageSize', 10); // Default pageSize = 10
        $orderBy = strtoupper($request->input('orderBy', 'DESC')); // Default sort = DESC

        // Ensure only valid order values
        if (!in_array($orderBy, ['ASC', 'DESC'])) {
            $orderBy = 'DESC';
        }

        $query = User::select('id', 'first_name', 'last_name', 'username', 'created_at', 'mobile', 'image', 'status', 'role', 'email');

        // Filter by first_name's first letter or email if 'search' is provided
        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('first_name', 'LIKE', $search . '%')
                    ->orWhere('last_name', 'LIKE', $search . '%')
                    ->orWhereRaw("CONCAT(first_name, ' ', last_name) LIKE ?", ["%$search%"]);
            });
        }

        $total = $query->count();

        $customers = $query->orderBy('created_at', $orderBy)
            ->skip(($page - 1) * $pageSize)
            ->take($pageSize)
            ->get()
            ->map(function ($user) {
                return [
                    'id' => $user->id,
                    'first_name' => $user->first_name ?: '',
                    'last_name' => $user->last_name ?: '',
                    'username' => $user->username ?: '',
                    'role' => $user->role ?: '',
                    'email' => $user->email ?: '',
                    'created_at' => $user->created_at ?: '',
                    'mobile' => $user->mobile ?: '',
                    'image' => $user->image
                        ? url('assets/images/users_images/' . $user->image)
                        : "",
                    'status' => $user->status,
                ];
            });

        return response()->json([
            'status' => true,
            'message' => 'All Users list fetched successfully',
            'allusers' => $customers,
            'Pagination' => [
                'total_pages' => ceil($total / $pageSize),
                'total_records' => $total,
                'current_page' => (int) $page,
                'records_per_page' => (int) $pageSize,
            ],
        ]);
    }







    // userStatusChange
    public function userStatusChange(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        $request->validate([
            'id' => 'required|exists:users,id',
            'status' => 'required|in:0,1',
        ]);

        $user = User::find($request->id);
        $user->status = $request->status;
        $user->save();

        return response()->json([
            'status' => true,
            'message' => 'Status changed successfully',
        ]);
    }


    public function categoryList(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        $search = $request->input('search');
        $page = $request->input('page', 1);
        $pageSize = $request->input('pageSize', 10);
        $orderBy = strtoupper($request->input('orderBy', 'DESC'));

        // Sanitize orderBy value
        if (!in_array($orderBy, ['ASC', 'DESC'])) {
            $orderBy = 'DESC';
        }

        $query = Category::select('id', 'category_name', 'category_image', 'category_icon', 'created_at')
            ->withCount('subcategories');

        // Apply search by first letter
        if ($search) {
            $query->where('category_name', 'LIKE', $search . '%');
        }

        $total = $query->count();

        // Apply order, skip, and limit
        $categories = $query->orderBy('created_at', $orderBy)
            ->skip(($page - 1) * $pageSize)
            ->take($pageSize)
            ->get();

        // Format image and calculate total services per category
        foreach ($categories as $category) {
            $category->category_image = $category->category_image
                ? url('assets/images/category_images/' . $category->category_image)
                : '';
            $category->category_icon = $category->category_icon
                ? url('assets/images/category_icons/' . $category->category_icon)
                : '';

            $category->total_services = Service::where('category_id', $category->id)->count();
        }

        return response()->json([
            'status' => true,
            'message' => $categories->isNotEmpty() ? 'Category found' : 'Category not found',
            'data' => $categories,
            'Pagination' => [
                'total_pages' => ceil($total / $pageSize),
                'total_records' => $total,
                'current_page' => (int) $page,
                'records_per_page' => (int) $pageSize,
            ],
        ]);
    }



    public function addCategory(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        $rules = [
            'category_name' => 'required',
            'category_image' => 'required',
        ];

        $customMessages = [
            'category_name.required' => 'Please enter the category name.',
            'category_image.required' => 'Please select the category image.',
        ];

        $validator = Validator::make($request->all(), $rules, $customMessages);

        if ($validator->fails()) {
            return response([
                'status' => false,
                'message' => $validator->errors()->first(),
            ], 422);
        }

        $category_image = null;
        $category_icon = null;

        if ($request->hasFile('category_image')) {
            $originalName = pathinfo($request->category_image->getClientOriginalName(), PATHINFO_FILENAME);
            $sanitizedFileName = Str::slug($originalName, '_') . '.' . $request->category_image->getClientOriginalExtension();
            $category_image = time() . '_' . $sanitizedFileName;
            $request->category_image->move(public_path('assets/images/category_images/'), $category_image);
        }

        if ($request->hasFile('category_icon')) {
            $originalName = pathinfo($request->category_icon->getClientOriginalName(), PATHINFO_FILENAME);
            $sanitizedFileName = Str::slug($originalName, '_') . '.' . $request->category_icon->getClientOriginalExtension();
            $category_icon = time() . '_' . $sanitizedFileName;
            $request->category_icon->move(public_path('assets/images/category_icons/'), $category_icon);
        }


        $category = new Category();
        $category->category_name = $request->category_name;
        $category->category_image = $category_image;
        $category->category_icon = $category_icon;
        $category->save();

        return response([
            'status' => true,
            'message' => 'Category added successfully!',
        ], 200);
    }

    // editCategory
    public function editCategory(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            $category_id = $request->id;

            if (!$category_id) {
                return response([
                    'status' => false,
                    'message' => 'Category ID is required.',
                ], 400);
            }

            $category = Category::find($category_id);

            if (!$category) {
                return response([
                    'status' => false,
                    'message' => 'Category not found.',
                ], 404);
            }

            // Update category name if provided
            if ($request->has('category_name')) {
                $category->category_name = $request->category_name;
            }

            // Handle image upload if provided
            if ($request->hasFile('category_image')) {
                $file = $request->file('category_image');
                $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                $file->move(public_path('assets/images/category_images'), $fileName);


                $category->category_image = $fileName;
            }

            if ($request->hasFile('category_icon')) {
                $file = $request->file('category_icon');
                $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                $file->move(public_path('assets/images/category_icons'), $fileName);


                $category->category_icon = $fileName;
            }

            $category->save();

            $categoryImageUrl = $category->category_image
                ? asset('assets/images/category_images/' . $category->category_image)
                : '';

            $categoryIconUrl = $category->category_image
                ? asset('assets/images/category_icons/' . $category->category_icon)
                : '';

            return response([
                'status' => true,
                'message' => 'Category updated successfully!',
                'category' => [
                    'id' => $category->id,
                    'category_name' => $category->category_name ?? '',
                    'category_image' => $categoryImageUrl,
                    'category_icon' => $categoryIconUrl,
                ],
            ], 200);
        } catch (\Exception $e) {
            return response([
                'status' => false,
                'message' => 'An error occurred while updating the category: ' . $e->getMessage(),
            ], 500);
        }
    }

    // deleteCategory
    public function deleteCategory(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $request->validate([
            'category_id' => 'required|exists:categories,id',
        ]);

        $category_id = $request->category_id;

        // Find the category
        $category = Category::find($category_id);

        if (!$category) {
            return response()->json([
                'status' => false,
                'message' => 'Category not found.',
            ], 404);
        }

        // Optionally delete the image from storage
        if ($category->category_image && file_exists(public_path('assets/images/category_images/' . $category->category_image))) {
            unlink(public_path('assets/images/category_images/' . $category->category_image));
        }

        $category->delete();

        return response()->json([
            'status' => true,
            'message' => 'Category deleted successfully.',
        ], 200);
    }

    public function subCategoryList(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $search = $request->input('search');
        $page = $request->input('page', 1);
        $pageSize = $request->input('pageSize', 10);
        $orderBy = strtoupper($request->input('orderBy', 'DESC'));

        if (!in_array($orderBy, ['ASC', 'DESC'])) {
            $orderBy = 'DESC';
        }

        $query = SubCategory::select(
            'sub_categories.id',
            'sub_categories.subcategory_name',
            'sub_categories.subcategory_image',
            'sub_categories.category_id',
            'categories.category_name',
            'sub_categories.created_at'
        )
            ->leftJoin('categories', 'sub_categories.category_id', '=', 'categories.id');

        if ($search) {
            $query->where('sub_categories.subcategory_name', 'LIKE', $search . '%');
        }

        $total = $query->count();

        $subcategories = $query->orderBy('sub_categories.created_at', $orderBy)
            ->skip(($page - 1) * $pageSize)
            ->take($pageSize)
            ->get();

        $subcategoryIds = $subcategories->pluck('id')->toArray();

        $total_services = 0;
        foreach ($subcategoryIds as $subId) {
            $total_services += Service::whereRaw("FIND_IN_SET(?, subcategory_id)", [$subId])->count();
        }

        $total_categories = SubCategory::distinct('category_id')->count('category_id');

        foreach ($subcategories as $subcategory) {
            $subcategory->subcategory_image = $subcategory->subcategory_image
                ? url('assets/images/subcategory_images/' . $subcategory->subcategory_image)
                : '';

            $subcategory->total_services = Service::whereRaw("FIND_IN_SET(?, subcategory_id)", [$subcategory->id])->count();
        }

        return response()->json([
            'status' => true,
            'message' => $subcategories->isNotEmpty() ? 'Sub Categories found' : 'Sub Categories not found',
            'data' => [
                'subcategories' => $subcategories,
                'total_categories' => $total_categories,
                'total_services' => $total_services,
            ],
            'Pagination' => [
                'total_pages' => ceil($total / $pageSize),
                'total_records' => $total,
                'current_page' => (int) $page,
                'records_per_page' => (int) $pageSize,
            ],
        ]);
    }



    // addSubCategory
    public function addSubCategory(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $rules = [
            'category_id' => 'required|exists:categories,id',
            'subcategory_name' => 'required',
            'subcategory_image' => 'required',
        ];

        $customMessages = [
            'category_id.required' => 'Please select a valid category.',
            'category_id.exists' => 'The selected category does not exist.',
            'subcategory_name.required' => 'Please enter the subcategory name.',
            'subcategory_image.required' => 'Please select a subcategory image.',
        ];

        $validator = Validator::make($request->all(), $rules, $customMessages);

        if ($validator->fails()) {
            return response([
                'status' => false,
                'message' => $validator->errors()->first(),
            ], 422);
        }

        // Upload image
        $subcategory_image = null;
        if ($request->hasFile('subcategory_image')) {
            $originalName = pathinfo($request->subcategory_image->getClientOriginalName(), PATHINFO_FILENAME);
            $sanitizedFileName = Str::slug($originalName, '_') . '.' . $request->subcategory_image->getClientOriginalExtension();
            $subcategory_image = time() . '_' . $sanitizedFileName;
            $request->subcategory_image->move(public_path('assets/images/subcategory_images/'), $subcategory_image);
        }

        // Save subcategory
        $subcategory = new SubCategory();
        $subcategory->category_id = $request->category_id;
        $subcategory->subcategory_name = $request->subcategory_name;
        $subcategory->subcategory_image = $subcategory_image;
        $subcategory->save();

        return response([
            'status' => true,
            'message' => 'Subcategory added successfully!',
        ], 200);
    }


    // editSubCategory
    public function editSubCategory(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            $subcategory_id = $request->id;

            if (!$subcategory_id) {
                return response([
                    'status' => false,
                    'message' => 'Subcategory ID is required.',
                ], 400);
            }

            $subcategory = SubCategory::find($subcategory_id);

            if (!$subcategory) {
                return response([
                    'status' => false,
                    'message' => 'Subcategory not found.',
                ], 404);
            }

            // Update subcategory name if provided
            if ($request->has('subcategory_name')) {
                $subcategory->subcategory_name = $request->subcategory_name;
            }

            // Update category_id if provided
            if ($request->has('category_id')) {
                $subcategory->category_id = $request->category_id;
            }

            // Handle subcategory image upload if provided
            if ($request->hasFile('subcategory_image')) {
                $file = $request->file('subcategory_image');
                $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                $file->move(public_path('assets/images/subcategory_images'), $fileName);

                $subcategory->subcategory_image = $fileName;
            }

            $subcategory->save();

            $subcategoryImageUrl = $subcategory->subcategory_image
                ? asset('assets/images/subcategory_images/' . $subcategory->subcategory_image)
                : '';

            return response([
                'status' => true,
                'message' => 'Subcategory updated successfully!',
                'subcategory' => [
                    'id' => $subcategory->id,
                    'subcategory_name' => $subcategory->subcategory_name ?? '',
                    'category_id' => $subcategory->category_id ?? '',
                    'subcategory_image' => $subcategoryImageUrl,
                ],
            ], 200);
        } catch (\Exception $e) {
            return response([
                'status' => false,
                'message' => 'An error occurred while updating the subcategory: ' . $e->getMessage(),
            ], 500);
        }
    }


    // deleteSubCategory
    public function deleteSubCategory(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $request->validate([
            'subcategory_id' => 'required|exists:sub_categories,id',
        ]);

        $subcategory_id = $request->subcategory_id;

        // Find the category
        $category = SubCategory::find($subcategory_id);

        if (!$category) {
            return response()->json([
                'status' => false,
                'message' => 'Category not found.',
            ], 404);
        }

        $category->delete();

        return response()->json([
            'status' => true,
            'message' => 'Sub Category deleted successfully.',
        ], 200);
    }



    public function serviceListAdmin_old(Request $request)
    {
        $search = $request->input('search');
        $page = (int) $request->input('page', 1);
        $pageSize = (int) $request->input('pageSize', 10);

        $orderColumn = $request->input('orderColumn', 'services.id');
        $orderDir = strtoupper($request->input('orderDir', 'DESC'));

        $allowedOrderColumns = [
            'services.id',
            'services.service_name',
            'services.created_at',
            'services.published_year',
            'categories.category_name',
        ];

        if (!in_array($orderColumn, $allowedOrderColumns)) {
            $orderColumn = 'services.id';
        }

        if (!in_array($orderDir, ['ASC', 'DESC'])) {
            $orderDir = 'DESC';
        }

        $query = Service::select(
            'services.id',
            'services.category_id',
            'services.service_name',
            'services.vendor_id',
            'services.address',
            'services.lat',
            'services.lon',
            'services.city',
            'services.published_year',
            'categories.category_name'
        )
            ->leftJoin('categories', 'services.category_id', '=', 'categories.id');

        if (!empty($search)) {
            $query->where('services.service_name', 'like', $search . '%');
        }

        $total = $query->count();

        $services = $query->orderBy($orderColumn, $orderDir)
            ->skip(($page - 1) * $pageSize)
            ->take($pageSize)
            ->get();

        foreach ($services as $service) {
            $images = ServiceImages::where('service_id', $service->id)->pluck('service_images')->toArray();
            $service->service_images = !empty($images)
                ? array_map(fn($img) => asset('assets/images/service_images/' . $img), $images)
                : [asset('assets/images/service_images/default_service.png')];
        }

        if ($services->isNotEmpty()) {
            return response()->json([
                'status' => true,
                'message' => 'Service List found',
                'ServiceList' => $services,
                'Pagination' => [
                    'total_pages' => ceil($total / $pageSize),
                    'total_records' => $total,
                    'current_page' => (int) $page,
                    'records_per_page' => (int) $pageSize,
                ],
            ]);
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Service List not found',
                'ServiceList' => [],
                'Pagination' => [
                    'total_pages' => ceil($total / $pageSize),
                    'total_records' => $total,
                    'current_page' => (int) $page,
                    'records_per_page' => (int) $pageSize,
                ],
            ]);
        }
    }

    public function serviceListAdmin(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $search = $request->input('search');
        $page = (int) $request->input('page', 1);
        $pageSize = (int) $request->input('pageSize', 10);

        $orderColumn = $request->input('orderColumn', 'services.id');
        $orderDir = strtoupper($request->input('orderDir', 'DESC'));

        $allowedOrderColumns = [
            'services.id',
            'services.service_name',
            'services.created_at',
            'services.published_year',
            'categories.category_name',
        ];

        if (!in_array($orderColumn, $allowedOrderColumns)) {
            $orderColumn = 'services.id';
        }

        if (!in_array($orderDir, ['ASC', 'DESC'])) {
            $orderDir = 'DESC';
        }

        $query = Service::select(
            'services.id',
            'services.category_id',
            'services.service_name',
            'services.vendor_id',
            'services.address',
            'services.lat',
            'services.lon',
            'services.city',
            'services.published_year',
            'services.created_at',
            'categories.category_name'
        )
            ->leftJoin('categories', 'services.category_id', '=', 'categories.id');

        if (!empty($search)) {
            $query->where('services.service_name', 'like', $search . '%');
        }

        $total = $query->count();

        $services = $query->orderBy($orderColumn, $orderDir)
            ->skip(($page - 1) * $pageSize)
            ->take($pageSize)
            ->get();

        foreach ($services as $service) {
            // 🔹 Service Images
            $images = ServiceImages::where('service_id', $service->id)->pluck('service_images')->toArray();
            $service->service_images = !empty($images)
                ? array_map(fn($img) => asset('assets/images/service_images/' . $img), $images)
                : [asset('assets/images/service_images/default_service.png')];

            // 🔹 Vendor Name
            $vendor = User::find($service->vendor_id);
            $service->vendor_name = $vendor ? $vendor->first_name . ' ' . $vendor->last_name : "";
            $service->email = $vendor ? $vendor->email : "";
            $service->image = $vendor && $vendor->image ? asset('assets/images/users_images/' . $vendor->image) : '';

            // 🔹 Average Review
            $avgReview = ServiceReview::where('service_id', $service->id)->avg('review_star');
            $service->avg_review = number_format($avgReview ?? 0, 1);

            // 🔹 Price Range (min - max)
            $prices = Stores::where('service_id', $service->id)->pluck('price');
            $minPrice = $prices->min() ?? 0;
            $maxPrice = $prices->max() ?? 0;

            $currency = SiteSetup::first()->default_currency ?? '$';
            $service->price_range = ($minPrice == $maxPrice)
                ? $currency . $minPrice
                : $currency . $minPrice . ' - ' . $currency . $maxPrice;
        }

        return response()->json([
            'status' => $services->isNotEmpty(),
            'message' => $services->isNotEmpty() ? 'Service List found' : 'Service List not found',
            'ServiceList' => $services,
            'Pagination' => [
                'total_pages' => ceil($total / $pageSize),
                'total_records' => $total,
                'current_page' => (int) $page,
                'records_per_page' => (int) $pageSize,
            ],
        ]);
    }

    public function serviceListWithoutPagination(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $search = $request->input('search');
        $orderColumn = $request->input('orderColumn', 'services.id');
        $orderDir = strtoupper($request->input('orderDir', 'DESC'));

        $allowedOrderColumns = [
            'services.id',
            'services.service_name',
            'services.created_at',
            'services.published_year',
            'categories.category_name',
        ];

        if (!in_array($orderColumn, $allowedOrderColumns)) {
            $orderColumn = 'services.id';
        }

        if (!in_array($orderDir, ['ASC', 'DESC'])) {
            $orderDir = 'DESC';
        }

        $query = Service::select(
            'services.id',
            'services.category_id',
            'services.service_name',
            'services.vendor_id',
            'services.address',
            'services.lat',
            'services.lon',
            'services.city',
            'services.published_year',
            'services.created_at',
            'categories.category_name'
        )
            ->leftJoin('categories', 'services.category_id', '=', 'categories.id');

        if (!empty($search)) {
            $query->where('services.service_name', 'like', $search . '%');
        }

        $services = $query->orderBy($orderColumn, $orderDir)->get();

        foreach ($services as $service) {
            $images = ServiceImages::where('service_id', $service->id)->pluck('service_images')->toArray();
            $service->service_images = !empty($images)
                ? array_map(fn($img) => asset('assets/images/service_images/' . $img), $images)
                : [asset('assets/images/service_images/default_service.png')];

            $vendor = User::find($service->vendor_id);
            $service->vendor_name = $vendor ? $vendor->first_name . ' ' . $vendor->last_name : "";
            $service->email = $vendor ? $vendor->email : "";
            $service->image = $vendor && $vendor->image ? asset('assets/images/users_images/' . $vendor->image) : '';

            $avgReview = ServiceReview::where('service_id', $service->id)->avg('review_star');
            $service->avg_review = number_format($avgReview ?? 0, 1);

            $prices = Stores::where('service_id', $service->id)->pluck('price');
            $minPrice = $prices->min() ?? 0;
            $maxPrice = $prices->max() ?? 0;

            $currency = SiteSetup::first()->default_currency ?? '$';
            $service->price_range = ($minPrice == $maxPrice)
                ? $currency . $minPrice
                : $currency . $minPrice . ' - ' . $currency . $maxPrice;
        }

        return response()->json([
            'status' => $services->isNotEmpty(),
            'message' => $services->isNotEmpty() ? 'Service List found' : 'Service List not found',
            'ServiceList' => $services,
        ]);
    }


    public function serviceListShowPagination(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $search = $request->input('search');
        $orderColumn = $request->input('orderColumn', 'services.id');
        $orderDir = strtoupper($request->input('orderDir', 'DESC'));

        $allowedOrderColumns = [
            'services.id',
            'services.service_name',
            'services.created_at',
            'services.published_year',
            'categories.category_name',
        ];

        if (!in_array($orderColumn, $allowedOrderColumns)) {
            $orderColumn = 'services.id';
        }

        if (!in_array($orderDir, ['ASC', 'DESC'])) {
            $orderDir = 'DESC';
        }

        $query = Service::select(
            'services.id',
            'services.category_id',
            'services.service_name',
            'services.vendor_id',
            'services.address',
            'services.lat',
            'services.lon',
            'services.city',
            'services.published_year',
            'services.created_at',
            'categories.category_name',
            DB::raw("
                CASE
                    WHEN MAX(goal_payment.service_id) IS NOT NULL
                    AND MAX(goal_payment.end_date_do) >= CURDATE()
                    THEN 1 ELSE 0
                END AS is_featured
            ")
        )
            ->leftJoin('categories', 'services.category_id', '=', 'categories.id')
            ->leftJoin('goal_payment', 'goal_payment.service_id', '=', 'services.id')
            ->groupBy(
                'services.id',
                'services.category_id',
                'services.service_name',
                'services.vendor_id',
                'services.address',
                'services.lat',
                'services.lon',
                'services.city',
                'services.published_year',
                'services.created_at',
                'categories.category_name'
            )
            ->havingRaw("MAX(goal_payment.end_date_do) >= CURDATE()");

        if (!empty($search)) {
            $query->where('services.service_name', 'like', $search . '%');
        }

        $services = $query->orderBy($orderColumn, $orderDir)->get();

        foreach ($services as $service) {
            $images = ServiceImages::where('service_id', $service->id)->pluck('service_images')->toArray();
            $service->service_images = !empty($images)
                ? array_map(fn($img) => asset('assets/images/service_images/' . $img), $images)
                : [asset('assets/images/service_images/default_service.png')];

            $vendor = User::find($service->vendor_id);
            $service->vendor_name = $vendor ? $vendor->first_name . ' ' . $vendor->last_name : "";
            $service->email = $vendor ? $vendor->email : "";
            $service->image = $vendor && $vendor->image ? asset('assets/images/users_images/' . $vendor->image) : '';

            $avgReview = ServiceReview::where('service_id', $service->id)->avg('review_star');
            $service->avg_review = number_format($avgReview ?? 0, 1);

            $prices = Stores::where('service_id', $service->id)->pluck('price');
            $minPrice = $prices->min() ?? 0;
            $maxPrice = $prices->max() ?? 0;

            $currency = SiteSetup::first()->default_currency ?? '$';
            $service->price_range = ($minPrice == $maxPrice)
                ? $currency . $minPrice
                : $currency . $minPrice . ' - ' . $currency . $maxPrice;
        }

        return response()->json([
            'status' => $services->isNotEmpty(),
            'message' => $services->isNotEmpty() ? 'Service List found' : 'Service List not found',
            'ServiceList' => $services,
        ]);
    }



    public function storeListAdminold(Request $request)
    {
        $search = $request->input('search');
        $page = (int) $request->input('page', 1);
        $pageSize = (int) $request->input('pageSize', 10);

        $orderColumn = $request->input('orderColumn', 'stores.id');
        $orderDir = strtoupper($request->input('orderDir', 'DESC'));

        $allowedOrderColumns = [
            'stores.id',
            // 'services.service_name',
            'stores.created_at',
            // 'services.published_year',
            // 'categories.category_name',
        ];

        if (!in_array($orderColumn, $allowedOrderColumns)) {
            $orderColumn = 'stores.id';
        }

        if (!in_array($orderDir, ['ASC', 'DESC'])) {
            $orderDir = 'DESC';
        }

        $query = Stores::select(
            'stores.id',
            'stores.subcategory_id',
            'stores.store_name',
            'services.vendor_id',
            'stores.store_description',
            'stores.price',
            'stores.service_id',
            'stores.created_at',
        )
            ->leftJoin('services', 'stores.service_id', '=', 'services.id');

        if (!empty($search)) {
            $query->where('stores.store_name', 'like', $search . '%');
        }

        $total = $query->count();

        $services = $query->orderBy($orderColumn, $orderDir)
            ->skip(($page - 1) * $pageSize)
            ->take($pageSize)
            ->get();

        foreach ($services as $service) {
            // 🔹 Service Images
            $images = ServiceImages::where('service_id', $service->id)->pluck('service_images')->toArray();
            $service->service_images = !empty($images)
                ? array_map(fn($img) => asset('assets/images/service_images/' . $img), $images)
                : [asset('assets/images/service_images/default_service.png')];

            // 🔹 Vendor Name
            $vendor = User::find($service->vendor_id);
            $service->vendor_name = $vendor ? $vendor->first_name . ' ' . $vendor->last_name : "";
            $service->email = $vendor ? $vendor->email : "";
            $service->image = $vendor && $vendor->image ? asset('assets/images/users_images/' . $vendor->image) : '';

            // 🔹 Average Review
            $avgReview = ServiceReview::where('service_id', $service->id)->avg('review_star');
            $service->avg_review = number_format($avgReview ?? 0, 1);

            // 🔹 Price Range (min - max)
            $prices = Stores::where('service_id', $service->id)->pluck('price');
            $minPrice = $prices->min() ?? 0;
            $maxPrice = $prices->max() ?? 0;

            $currency = SiteSetup::first()->default_currency ?? '$';
            $service->price_range = ($minPrice == $maxPrice)
                ? $currency . $minPrice
                : $currency . $minPrice . ' - ' . $currency . $maxPrice;
        }

        return response()->json([
            'status' => $services->isNotEmpty(),
            'message' => $services->isNotEmpty() ? 'Service List found' : 'Service List not found',
            'ServiceList' => $services,
            'Pagination' => [
                'total_pages' => ceil($total / $pageSize),
                'total_records' => $total,
                'current_page' => (int) $page,
                'records_per_page' => (int) $pageSize,
            ],
        ]);
    }

    public function storeListAdmin(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        $search = $request->input('search');
        $page = (int) $request->input('page', 1);
        $pageSize = (int) $request->input('pageSize', 10);

        $orderColumn = $request->input('orderColumn', 'stores.id');
        $orderDir = strtoupper($request->input('orderDir', 'DESC'));

        $allowedOrderColumns = [
            'stores.id',
            'stores.created_at',
        ];

        if (!in_array($orderColumn, $allowedOrderColumns)) {
            $orderColumn = 'stores.id';
        }

        if (!in_array($orderDir, ['ASC', 'DESC'])) {
            $orderDir = 'DESC';
        }

        $query = Stores::select(
            'stores.id',
            // 'stores.subcategory_id',
            'stores.store_name',
            'services.vendor_id',
            'stores.store_description',
            'stores.price',
            'stores.service_id',
            'stores.created_at',
            'services.service_name',
        )
            ->leftJoin('services', 'stores.service_id', '=', 'services.id');

        if (!empty($search)) {
            $query->where('stores.store_name', 'like', $search . '%');
        }

        $total = $query->count();

        $services = $query->orderBy($orderColumn, $orderDir)
            ->skip(($page - 1) * $pageSize)
            ->take($pageSize)
            ->get();

        $currency = SiteSetup::first()->default_currency ?? '$';

        foreach ($services as $service) {

            $service->service_id = $service->service_id ? (int) $service->service_id : 0;
            // 🔹 Store Images (all)
            $images = StoreImages::where('store_id', $service->id)->pluck('store_images')->toArray();
            $service->store_images = !empty($images)
                ? array_map(fn($img) => asset('assets/images/store_images/' . $img), $images)
                : [asset('assets/images/store_images/default_store.png')];

            // 🔹 Vendor Name & Info
            $vendor = User::find($service->vendor_id);
            $service->vendor_name = $vendor ? $vendor->first_name . ' ' . $vendor->last_name : "";
            $service->email = $vendor ? $vendor->email : "";
            $service->image = $vendor && $vendor->image ? asset('assets/images/users_images/' . $vendor->image) : '';
            $service->service_name = $service->service_name ? $service->service_name : "";


            $service->price = $currency . $service->price;


            // 🔹 Price Range
            // $prices = Store::where('service_id', $service->service_id)->pluck('price');
            // $minPrice = $prices->min() ?? 0;
            // $maxPrice = $prices->max() ?? 0;

            // $service->price_range = ($minPrice == $maxPrice)
            //     ? $currency . $minPrice
            //     : $currency . $minPrice . ' - ' . $currency . $maxPrice;
        }

        return response()->json([
            'status' => $services->isNotEmpty(),
            'message' => $services->isNotEmpty() ? 'Service List found' : 'Service List not found',
            'ServiceList' => $services,
            'Pagination' => [
                'total_pages' => ceil($total / $pageSize),
                'total_records' => $total,
                'current_page' => (int) $page,
                'records_per_page' => (int) $pageSize,
            ],
        ]);
    }





    // deleteService
    public function deleteService(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $request->validate([
            'service_id' => 'required|exists:services,id',
        ]);

        $service_id = $request->service_id;

        // Find the service
        $service = Service::find($service_id);

        if (!$service) {
            return response()->json([
                'status' => false,
                'message' => 'Service not found.',
            ], 404);
        }

        // Delete related records
        ServiceImages::where('service_id', $service_id)->delete();
        ServiceReview::where('service_id', $service_id)->delete();
        ServiceLike::where('service_id', $service_id)->delete();

        // Delete the service
        $service->delete();

        return response()->json([
            'status' => true,
            'message' => 'Service deleted successfully.',
        ], 200);
    }

    public function ReviewList(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $search = $request->input('search');
        $page = $request->input('page', 1);
        $pageSize = $request->input('pageSize', 10);
        $orderBy = strtoupper($request->input('orderBy', 'DESC'));

        if (!in_array($orderBy, ['ASC', 'DESC'])) {
            $orderBy = 'DESC';
        }

        $query = ServiceReview::select(
            'service_review.id',
            'service_review.user_id',
            'service_review.service_id',
            'service_review.review_star',
            'service_review.review_message',
            'service_review.created_at',
            'services.service_name',
            'categories.category_name',
            'users.first_name',
            'users.last_name',
            'users.email',
            'users.image as user_image'
        )
            ->leftJoin('services', 'service_review.service_id', '=', 'services.id')
            ->leftJoin('categories', 'services.category_id', '=', 'categories.id')
            ->leftJoin('users', 'service_review.user_id', '=', 'users.id');

        if ($search) {
            $query->where('services.service_name', 'like', $search . '%');
        }

        $total = $query->count();

        $reviewlist = $query
            ->orderBy('service_review.created_at', $orderBy)
            ->skip(($page - 1) * $pageSize)
            ->take($pageSize)
            ->get();

        foreach ($reviewlist as $item) {
            $item->created_at = $item->created_at;

            // Fetch and format service images
            $service_images = ServiceImages::where('service_id', $item->service_id)
                ->pluck('service_images')
                ->toArray();

            $item->service_images = !empty($service_images)
                ? array_map(function ($image) {
                    return asset('assets/images/service_images/' . $image);
                }, $service_images)
                : [asset('assets/images/service_images/default_service.png')];

            // Format user image path
            $item->user_image = !empty($item->user_image)
                ? asset('assets/images/users_images/' . $item->user_image)
                : "";

            $item->service_name = $item->service_name ?? '';
            $item->category_name = $item->category_name ?? '';
        }

        return response()->json([
            'status' => $reviewlist->isNotEmpty(),
            'message' => $reviewlist->isNotEmpty() ? 'Review list found' : 'Review list not found',
            'reviewlist' => $reviewlist,
            'Pagination' => [
                'total_pages' => ceil($total / $pageSize),
                'total_records' => $total,
                'current_page' => (int) $page,
                'records_per_page' => (int) $pageSize,
            ],
        ]);
    }


    // addSubscription
    public function addSubscription(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $rules = [
            'plan_name' => 'required',
            'description' => 'required',
            'price' => 'required',
            'subtext' => 'required',
            'duration' => 'required',
        ];

        $customMessages = [
            'plan_name.required' => 'Please enter the plan name.',
            'description.required' => 'Please enter the description.',
            'price.required' => 'Please enter the price.',
            'subtext.required' => 'Please enter the subtext.',
            'duration.required' => 'Please enter the duration.',
        ];

        $validator = Validator::make($request->all(), $rules, $customMessages);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'message' => $validator->errors()->first(),
            ], 422);
        }

        $plan = new SubscriptionPlan();
        $plan->plan_name = $request->plan_name;
        $plan->description = $request->description;
        $plan->price = $request->price;
        $plan->subtext = $request->subtext;
        $plan->duration = $request->duration;
        $plan->save();

        return response()->json([
            'status' => true,
            'message' => 'Subscription plan added successfully!',
        ], 200);
    }


    // editSubscription
    public function editSubscription(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            $subscription_id = $request->id;

            if (!$subscription_id) {
                return response()->json([
                    'status' => false,
                    'message' => 'Subscription ID is required.',
                ], 400);
            }

            $subscription = SubscriptionPlan::find($subscription_id);

            if (!$subscription) {
                return response()->json([
                    'status' => false,
                    'message' => 'Subscription not found.',
                ], 404);
            }

            // Update fields if present in the request
            if ($request->has('plan_name')) {
                $subscription->plan_name = $request->plan_name;
            }

            if ($request->has('description')) {
                $subscription->description = $request->description;
            }

            if ($request->has('price')) {
                $subscription->price = $request->price;
            }

            if ($request->has('subtext')) {
                $subscription->subtext = $request->subtext;
            }

            if ($request->has('duration')) {
                $subscription->duration = $request->duration;
            }

            $subscription->save();

            return response()->json([
                'status' => true,
                'message' => 'Subscription updated successfully!',
                'subscription' => [
                    'id' => $subscription->id,
                    'plan_name' => $subscription->plan_name ?? '',
                    'description' => $subscription->description ?? '',
                    'price' => $subscription->price ?? '',
                    'subtext' => $subscription->subtext ?? '',
                    'duration' => $subscription->duration ?? '',
                ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating the subscription: ' . $e->getMessage(),
            ], 500);
        }
    }


    // deleteSubscription
    public function deleteSubscription(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $request->validate([
            'subscription_id' => 'required|exists:subscription_plan,id',
        ], [
            'subscription_id.required' => 'Subscription ID is required.',
            'subscription_id.exists' => 'The selected subscription does not exist.',
        ]);

        $subscription_id = $request->subscription_id;

        $subscription = SubscriptionPlan::find($subscription_id);

        if (!$subscription) {
            return response()->json([
                'status' => false,
                'message' => 'Subscription not found.',
            ], 404);
        }

        $subscription->delete();

        return response()->json([
            'status' => true,
            'message' => 'Subscription deleted successfully.',
        ], 200);
    }


    // subscribedUsers
    public function subscribedUsersold(Request $request)
    {
        $customers = UserSubscription::select(
            'id',
            'user_id',
            'plan_name',
            'created_at',
            'price',
            'start_date',
            'expire_date',
            'payment_mode',
            'status'
        )
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($user) {
                return [
                    'id' => $user->id,
                    'user_id' => $user->user_id,
                    'plan_name' => $user->plan_name ?? '',
                    'created_at' => $user->created_at
                        ? Carbon::parse($user->created_at)->format('d M, Y / g:i A')
                        : '',
                    'price' => $user->price ?? '',
                    'start_date' => $user->start_date ?? '',
                    'expire_date' => $user->expire_date ?? '',
                    'payment_mode' => $user->payment_mode ?? '',
                    'status' => $user->status,
                ];
            });

        return response()->json([
            'status' => true,
            'message' => 'User subscription list fetched successfully.',
            'customers' => $customers
        ]);
    }
    public function subscribedUsers(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        $search = $request->input('search');
        $page = (int) $request->input('page', 1);
        $pageSize = (int) $request->input('pageSize', 10);
        $orderColumn = $request->input('orderColumn', 'created_at');
        $orderDir = strtoupper($request->input('orderDir', 'DESC'));

        // Allowed ordering columns
        $allowedOrderColumns = [
            'id',
            'user_id',
            'plan_name',
            'created_at',
            'price',
            'start_date',
            'expire_date',
            'payment_mode',
            'status'
        ];

        if (!in_array($orderColumn, $allowedOrderColumns)) {
            $orderColumn = 'created_at';
        }

        if (!in_array($orderDir, ['ASC', 'DESC'])) {
            $orderDir = 'DESC';
        }

        // $query = UserSubscription::select(
        //     'id',
        //     'user_id',
        //     'plan_name',
        //     'created_at',
        //     'price',
        //     'start_date',
        //     'expire_date',
        //     'payment_mode',
        //     'status'
        // );

        $query = UserSubscription::with('user:id,first_name,last_name,email,image') // eager load user
            ->select(
                'id',
                'user_id',
                'plan_name',
                'created_at',
                'price',
                'start_date',
                'expire_date',
                'payment_mode',
                'status'
            );

        if (!empty($search)) {
            $query->whereHas('user', function ($q) use ($search) {
                $q->whereRaw("CONCAT(first_name, ' ', last_name) LIKE ?", ["%$search%"])
                    ->orWhere('first_name', 'like', '%' . $search . '%')
                    ->orWhere('last_name', 'like', '%' . $search . '%')
                    ->orWhere('email', 'like', '%' . $search . '%');
            });
        }

        $total = $query->count();

        $subscriptions = $query->orderBy($orderColumn, $orderDir)
            ->skip(($page - 1) * $pageSize)
            ->take($pageSize)
            ->get()
            ->map(function ($user) {
                $vendor = $user->user; // from eager loading
                return [
                    'id' => $user->id,
                    'user_id' => $user->user_id,
                    'plan_name' => $user->plan_name ?? '',
                    'created_at' => $user->created_at
                        ? Carbon::parse($user->created_at)->format('d M, Y / g:i A')
                        : '',
                    'price' => $user->price ?? '',
                    'start_date' => $user->start_date ?? '',
                    'expire_date' => $user->expire_date ?? '',
                    'payment_mode' => $user->payment_mode ?? '',
                    'status' => $user->status,

                    // Vendor info
                    'vendor_name' => $vendor ? $vendor->first_name . ' ' . $vendor->last_name : '',
                    'email' => $vendor ? $vendor->email : '',
                    'image' => $vendor && $vendor->image
                        ? asset('assets/images/users_images/' . $vendor->image)
                        : "",
                ];
            });

        return response()->json([
            'status' => $subscriptions->isNotEmpty(),
            'message' => $subscriptions->isNotEmpty()
                ? 'User subscription list fetched successfully.'
                : 'No subscriptions found.',
            'customers' => $subscriptions,
            'Pagination' => [
                'total_pages' => ceil($total / $pageSize),
                'total_records' => $total,
                'current_page' => $page,
                'records_per_page' => $pageSize,
            ],
        ]);
    }

    public function viewSubscriberUser(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $id = $request->input('id');



        if (!$id) {

            return response([

                'status' => false,

                'message' => 'Subsciber ID is required.',

            ], 400);
        }

        $category = UserSubscription::findOrFail($id);

        $vendor = User::find($category->user_id);
        $category->vendor_name = $vendor ? $vendor->first_name . ' ' . $vendor->last_name : "";
        $category->email = $vendor ? $vendor->email : "";
        $category->image = $vendor && $vendor->image ? asset('assets/images/users_images/' . $vendor->image) : '';
        $category->mobile = $vendor ? $vendor->mobile : "";




        // // Fetch subcategories related to this category

        // $subcategories = SubCategory::where('category_id', $cat_id)

        //     ->withCount('services') // Assuming there's a services relationship in SubCategory

        //     ->get();



        // Attach full image URL to each subcategory

        // foreach ($subcategories as $subcategory) {

        //     $subcategory->subcategory_image = $subcategory->subcategory_image

        //         ? url('assets/images/subcategory_images/' . $subcategory->subcategory_image)

        //         : url('assets/images/subcategory_images/default_subcategory.png');
        // }



        return response([

            'status' => true,

            'message' => 'Subsciber User View successfully!',

            'category' => $category,

            // 'subcategories' => $subcategories,

        ], 200);
    }



    // sponserPayment
    public function sponserPaymentold(Request $request)
    {
        // Get all goal payments
        $goalPayments = GoalPayment::with('goal') // Assumes a relationship exists
            ->orderBy('created_at', 'desc')
            ->get();

        if ($goalPayments->isEmpty()) {
            return response()->json([
                'status' => false,
                'message' => 'No goal payments found',
                'GoalData' => [],
            ]);
        }

        $data = $goalPayments->map(function ($payment) {
            $goal = $payment->goal;

            return [
                'goal_payment_id' => $payment->id,
                'vendor_id' => $payment->vendor_id,
                'goal_id' => $payment->goal_id,
                'payment_amount' => $payment->amount ?? '',
                'payment_date' => $payment->created_at ? $payment->created_at->format('d M, Y') : '',
                'goal_name' => $goal->goal_name ?? '',
                'start_date' => $goal->start_date ?? '',
                'end_date' => $goal->end_date ?? '',
                'days' => $goal->days ?? '',
                'goal_price' => $goal->price ?? '',
            ];
        });

        return response()->json([
            'status' => true,
            'message' => 'Goal payments fetched successfully',
            'GoalData' => $data,
        ]);
    }

    public function sponserPayment(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $search = $request->input('search');
        $page = (int) $request->input('page', 1);
        $pageSize = (int) $request->input('pageSize', 10);
        $orderColumn = $request->input('orderColumn', 'goal_payment.created_at');
        $orderDir = strtoupper($request->input('orderDir', 'DESC'));

        $allowedOrderColumns = [
            'goal_payment.id',
            'goal_payment.vendor_id',
            'goal_payment.goal_id',
            'goal_payment.price',
            'goal_payment.created_at',
            'goal_payment.payment_mode',
        ];

        if (!in_array($orderColumn, $allowedOrderColumns)) {
            $orderColumn = 'goal_payment.created_at';
        }

        if (!in_array($orderDir, ['ASC', 'DESC'])) {
            $orderDir = 'DESC';
        }

        $query = GoalPayment::with(['goal.campaign', 'service'])
            ->select('goal_payment.*')
            ->orderBy($orderColumn, $orderDir);

        // if (!empty($search)) {
        //     $query->whereHas('goal', function ($q) use ($search) {
        //         $q->where('goal_name', 'like', '%' . $search . '%');
        //     });
        // }

        if (!empty($search)) {
            $query->where(function ($q) use ($search) {
                // Remove goal->goal_name search
                $q->orWhereHas('goal.campaign', function ($campaignQuery) use ($search) {
                    $campaignQuery->where('campaign_name', 'like', '%' . $search . '%');
                })->orWhereHas('service', function ($serviceQuery) use ($search) {
                    $serviceQuery->where('service_name', 'like', '%' . $search . '%');
                });
            });
        }

        $total = $query->count();

        $goalPayments = $query
            ->skip(($page - 1) * $pageSize)
            ->take($pageSize)
            ->get();

        if ($goalPayments->isEmpty()) {
            return response()->json([
                'status' => false,
                'message' => 'No goal payments found',
                'GoalData' => [],
            ]);
        }

        $data = $goalPayments->map(function ($payment) {
            $goal = $payment->goal;
            $campaign = $goal->campaign ?? null;
            $service = $payment->service;

            $time_stamp = now()->format('Y-m-d');

            $done = GoalPayment::where('goal_payment.id', $payment->id)->where('goal_payment.end_date_do', '>=', $time_stamp)->first();



            return [
                'goal_payment_id' => $payment->id,
                'vendor_id' => $payment->vendor_id,
                'goal_id' => $payment->goal_id,
                'payment_amount' => $payment->price ?? '',
                'payment_date' => $payment->created_at ? $payment->created_at->format('d M, Y') : '',
                'goal_name' => $goal->goal_name ?? '',
                'start_date' => $goal->start_date ?? '',
                'end_date' => $goal->end_date ?? '',
                'days' => $goal->days ?? '',
                'goal_price' => $payment->price ?? '',
                'campaign_name' => $campaign->campaign_name ?? '',
                'service_name' => $service->service_name ?? '',
                'payment_mode' => $payment->payment_mode ?? '',
                'created_at' => $goal->created_at ?? '',
                'all_status' => $done ? "1" : "0",

            ];
        });

        return response()->json([
            'status' => true,
            'message' => 'Goal payments fetched successfully',
            'GoalData' => $data,
            'Pagination' => [
                'total_pages' => ceil($total / $pageSize),
                'total_records' => $total,
                'current_page' => $page,
                'records_per_page' => $pageSize,
            ],
        ]);
    }

    public function viewsponserPayment(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $goal_payment_id = $request->input('goal_payment_id');

        if (!$goal_payment_id) {
            return response([
                'status' => false,
                'message' => 'Goal Payment Id is required.',
            ], 400);
        }

        // Fetch the GoalPayment with relations
        $payment = GoalPayment::with(['goal.campaign', 'service', 'vendor'])->find($goal_payment_id);

        if (!$payment) {
            return response([
                'status' => false,
                'message' => 'Goal Payment not found.',
            ], 404);
        }

        $goal = $payment->goal;
        $campaign = $goal->campaign ?? null;
        $service = $payment->service;
        $vendor = $payment->vendor;

        $data = [
            'goal_payment_id' => $payment->id,
            'vendor_id' => $payment->vendor_id,
            'goal_id' => $payment->goal_id,
            'payment_amount' => $payment->price ?? '',
            'payment_date' => $payment->created_at ? $payment->created_at->format('d M, Y') : '',
            'goal_name' => $goal->goal_name ?? '',
            'start_date' => $goal->start_date ?? '',
            'end_date' => $goal->end_date ?? '',
            'days' => $goal->days ?? '',
            'goal_price' => $payment->price ?? '',
            'campaign_name' => $campaign->campaign_name ?? '',
            'service_name' => $service->service_name ?? '',
            'payment_mode' => $payment->payment_mode ?? '',
            'created_at' => $goal ? $goal->created_at : '',
            'vendor_name' => $vendor ? $vendor->first_name . ' ' . $vendor->last_name : '',
            'vendor_email' => $vendor->email ?? '',
            'vendor_mobile' => $vendor->mobile ?? '',
            'vendor_image' => $vendor && $vendor->image
                ? url('/assets/images/users_images/' . $vendor->image)
                : url('assets/images/usesr_images/default_user.png'),
        ];

        return response([
            'status' => true,
            'message' => 'Goal Payment View successfully!',
            'view_sponser' => $data,
        ], 200);
    }








    // sliderList
    public function sliderListold(Request $request)
    {
        $customers = Slides::select('id', 'description', 'image', 'status')
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($user) {
                return [
                    'id' => $user->id,
                    'description' => $user->description ?: '',
                    'status' => $user->status,
                    'image' => $user->image
                        ? url('assets/images/slides/' . $user->image)
                        : '',
                ];
            });

        return response()->json([
            'status' => true,
            'message' => 'Slider list fetched successfully',
            'sliders' => $customers
        ]);
    }

    public function sliderList(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $search = $request->input('search');
        $page = (int) $request->input('page', 1);
        $pageSize = (int) $request->input('pageSize', 10);
        $orderColumn = $request->input('orderColumn', 'created_at');
        $orderDir = strtoupper($request->input('orderDir', 'DESC'));

        $allowedOrderColumns = ['id', 'created_at', 'status'];

        if (!in_array($orderColumn, $allowedOrderColumns)) {
            $orderColumn = 'created_at';
        }

        if (!in_array($orderDir, ['ASC', 'DESC'])) {
            $orderDir = 'DESC';
        }

        $query = Slides::select('id', 'description', 'image', 'status', 'created_at');

        if (!empty($search)) {
            $query->where('description', 'like', '%' . $search . '%');
        }

        $total = $query->count();

        $customers = $query->orderBy($orderColumn, $orderDir)
            ->skip(($page - 1) * $pageSize)
            ->take($pageSize)
            ->get()
            ->map(function ($user) {
                return [
                    'id' => $user->id,
                    'description' => $user->description ?: '',
                    'status' => $user->status,
                    'image' => $user->image
                        ? url('assets/images/slides/' . $user->image)
                        : '',
                    'created_at' => $user->created_at,
                ];
            });

        return response()->json([
            'status' => true,
            'message' => 'Slider list fetched successfully',
            'sliders' => $customers,
            'Pagination' => [
                'total_pages' => ceil($total / $pageSize),
                'total_records' => $total,
                'current_page' => $page,
                'records_per_page' => $pageSize,
            ]
        ]);
    }

    public function updateSliderStatus(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        try {
            $subscription_id = $request->id;

            if (!$subscription_id) {
                return response()->json([
                    'status' => false,
                    'message' => 'Slider ID is required.',
                ], 400);
            }

            $subscription = Slides::find($subscription_id);

            if (!$subscription) {
                return response()->json([
                    'status' => false,
                    'message' => 'Slider ID not found.',
                ], 404);
            }

            // Update fields if present in the request
            if ($request->has('status')) {
                $subscription->status = $request->status;
            }

            $subscription->save();

            return response()->json([
                'status' => true,
                'message' => 'Slider Status updated successfully!',
                // 'subscription' => [
                //     'id' => $subscription->id,
                //     'plan_name' => $subscription->plan_name ?? '',
                //     'description' => $subscription->description ?? '',
                //     'price' => $subscription->price ?? '',
                //     'subtext' => $subscription->subtext ?? '',
                // ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating the subscription: ' . $e->getMessage(),
            ], 500);
        }
    }






    // addSlider
    public function addSlider(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        $rules = [
            'description' => 'required',
            'image' => 'required',
            'status' => 'required',
        ];

        $customMessages = [
            'description.required' => 'Please enter the description.',
            'image.required' => 'Please select the image.',
            'status.required' => 'Please enter the status.',
        ];

        $validator = Validator::make($request->all(), $rules, $customMessages);

        if ($validator->fails()) {
            return response([
                'status' => false,
                'message' => $validator->errors()->first(),
            ], 422);
        }

        $image = null;

        if ($request->hasFile('image')) {
            $originalName = pathinfo($request->image->getClientOriginalName(), PATHINFO_FILENAME);
            $sanitizedFileName = Str::slug($originalName, '_') . '.' . $request->image->getClientOriginalExtension();
            $image = time() . '_' . $sanitizedFileName;
            $request->image->move(public_path('assets/images/slides/'), $image);
        }

        $category = new Slides();
        $category->description = $request->description;
        $category->status = $request->status;
        $category->image = $image;
        $category->save();

        return response([
            'status' => true,
            'message' => 'Slides added successfully!',
        ], 200);
    }





    // deleteSlider
    public function deleteSlider(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        $request->validate([
            'slider_id' => 'required|exists:slides,id',
        ]);

        $slider_id = $request->slider_id;

        // Find the category
        $category = Slides::find($slider_id);

        if (!$category) {
            return response()->json([
                'status' => false,
                'message' => 'Slider not found.',
            ], 404);
        }

        $category->delete();

        return response()->json([
            'status' => true,
            'message' => 'Slider deleted successfully.',
        ], 200);
    }



    // currencyList
    public function currencyListold(Request $request)
    {
        $customers = Currencies::select('id', 'currency', 'name')
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($user) {
                return [
                    'id' => $user->id,
                    'currency' => $user->currency ?: '',
                    'name' => $user->name ?: '',
                ];
            });

        return response()->json([
            'status' => true,
            'message' => 'Currencies fetched successfully',
            'currencies' => $customers
        ]);
    }

    public function currencyList(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $search = $request->input('search');
        $page = (int) $request->input('page', 1);
        $pageSize = (int) $request->input('pageSize', 10);
        $orderColumn = $request->input('orderColumn', 'created_at');
        $orderDir = strtoupper($request->input('orderDir', 'DESC'));

        $allowedOrderColumns = ['id', 'currency', 'name', 'created_at'];

        if (!in_array($orderColumn, $allowedOrderColumns)) {
            $orderColumn = 'created_at';
        }

        if (!in_array($orderDir, ['ASC', 'DESC'])) {
            $orderDir = 'DESC';
        }

        $query = Currencies::select('id', 'currency', 'name', 'created_at');

        if (!empty($search)) {
            $query->where(function ($q) use ($search) {
                $q->where('currency', 'like', '%' . $search . '%')
                    ->orWhere('name', 'like', '%' . $search . '%');
            });
        }

        $total = $query->count();

        $customers = $query->orderBy($orderColumn, $orderDir)
            ->skip(($page - 1) * $pageSize)
            ->take($pageSize)
            ->get()
            ->map(function ($user) {
                return [
                    'id' => $user->id,
                    'currency' => $user->currency ?: '',
                    'name' => $user->name ?: '',
                    'created_at' => $user->created_at ?: '',
                ];
            });

        return response()->json([
            'status' => true,
            'message' => 'Currencies fetched successfully',
            'currencies' => $customers,
            'Pagination' => [
                'total_pages' => ceil($total / $pageSize),
                'total_records' => $total,
                'current_page' => $page,
                'records_per_page' => $pageSize,
            ]
        ]);
    }

    public function faqList(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $search = $request->input('search');
        $page = (int) $request->input('page', 1);
        $pageSize = (int) $request->input('pageSize', 10);

        $query = Faq::select('id', 'question', 'answer');

        if (!empty($search)) {
            $query->where(function ($q) use ($search) {
                $q->where('question', 'like', '%' . $search . '%')
                    ->orWhere('answer', 'like', '%' . $search . '%');
            });
        }

        $total = $query->count();

        $termcondition = $query->orderBy('created_at', 'desc')
            ->skip(($page - 1) * $pageSize)
            ->take($pageSize)
            ->get();

        if ($termcondition->isNotEmpty()) {
            return response()->json([
                'status' => true,
                'message' => 'Faq found',
                'data' => $termcondition,
                'Pagination' => [
                    'total_pages' => ceil($total / $pageSize),
                    'total_records' => $total,
                    'current_page' => $page,
                    'records_per_page' => $pageSize,
                ]
            ]);
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Faq not found',
            ]);
        }
    }




    // addCurrency
    public function addCurrency(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        $rules = [
            'currency' => 'required',
            'name' => 'required',
        ];

        $customMessages = [
            'currency.required' => 'Please enter the currency.',
            'name.required' => 'Please enter the name.',
        ];

        $validator = Validator::make($request->all(), $rules, $customMessages);

        if ($validator->fails()) {
            return response([
                'status' => false,
                'message' => $validator->errors()->first(),
            ], 422);
        }

        $category = new Currencies();
        $category->currency = $request->currency;
        $category->name = $request->name;
        $category->save();

        return response([
            'status' => true,
            'message' => 'Currency added successfully!',
        ], 200);
    }


    // deleteCurrency
    public function deleteCurrency(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        $request->validate([
            'currency_id' => 'required|exists:currencies,id',
        ]);

        $currency_id = $request->currency_id;

        // Find the category
        $category = Currencies::find($currency_id);

        if (!$category) {
            return response()->json([
                'status' => false,
                'message' => 'Currencies not found.',
            ], 404);
        }

        $category->delete();

        return response()->json([
            'status' => true,
            'message' => 'Currencies deleted successfully.',
        ], 200);
    }



    //  refundPolicy
    public function refundPolicyold(Request $request)
    {
        $privacypolicy = RefundPolicy::select('id', 'text')->get();

        if ($privacypolicy->isNotEmpty()) {
            return response()->json([
                'status' => true,
                'message' => 'Refund policy found',
                'data' => $privacypolicy,
            ]);
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Refund policy not found',
            ]);
        }
    }

    public function refundPolicy(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            // Always update ID = 1
            $policy = RefundPolicy::find(1);

            if (!$policy) {
                return response()->json([
                    'status' => false,
                    'message' => 'Refund policy not found.',
                ], 404);
            }

            // Update text if provided
            if ($request->has('text')) {
                $policy->text = $request->text;
            }

            $policy->save();

            return response()->json([
                'status' => true,
                'message' => 'Refund policy updated successfully!',
                'data' => [
                    'id' => $policy->id,
                    'text' => $policy->text ?? '',
                ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating the refund policy: ' . $e->getMessage(),
            ], 500);
        }
    }




    //  cancelPolicy
    public function cancelPolicyold(Request $request)
    {
        $privacypolicy = CancelPolicy::select('id', 'text')->get();

        if ($privacypolicy->isNotEmpty()) {
            return response()->json([
                'status' => true,
                'message' => 'Cancel policy found',
                'data' => $privacypolicy,
            ]);
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Cancel policy not found',
            ]);
        }
    }


    public function cancelPolicy(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            // Always update ID = 1
            $policy = CancelPolicy::find(1);

            if (!$policy) {
                return response()->json([
                    'status' => false,
                    'message' => 'Cancel policy not found.',
                ], 404);
            }

            // Update text if provided
            if ($request->has('text')) {
                $policy->text = $request->text;
            }

            $policy->save();

            return response()->json([
                'status' => true,
                'message' => 'Cancel policy updated successfully!',
                'data' => [
                    'id' => $policy->id,
                    'text' => $policy->text ?? '',
                ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating the refund policy: ' . $e->getMessage(),
            ], 500);
        }
    }




    //  about
    public function aboutold(Request $request)
    {
        $privacypolicy = About::select('id', 'text')->get();

        if ($privacypolicy->isNotEmpty()) {
            return response()->json([
                'status' => true,
                'message' => 'About policy found',
                'data' => $privacypolicy,
            ]);
        } else {
            return response()->json([
                'status' => false,
                'message' => 'About policy not found',
            ]);
        }
    }

    public function about(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        try {
            // Always update ID = 1
            $policy = About::find(1);

            if (!$policy) {
                return response()->json([
                    'status' => false,
                    'message' => 'About policy not found.',
                ], 404);
            }

            // Update text if provided
            if ($request->has('text')) {
                $policy->text = $request->text;
            }

            $policy->save();

            return response()->json([
                'status' => true,
                'message' => 'About policy updated successfully!',
                'data' => [
                    'id' => $policy->id,
                    'text' => $policy->text ?? '',
                ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating the refund policy: ' . $e->getMessage(),
            ], 500);
        }
    }



    //  contactUs
    public function contactUsold(Request $request)
    {
        $privacypolicy = ContactUs::select('id', 'text')->get();

        if ($privacypolicy->isNotEmpty()) {
            return response()->json([
                'status' => true,
                'message' => 'Contact US found',
                'data' => $privacypolicy,
            ]);
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Contact US  not found',
            ]);
        }
    }

    public function privacyPolicy(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            // Always update ID = 1
            $policy = PrivacyPolicy::find(1);

            if (!$policy) {
                return response()->json([
                    'status' => false,
                    'message' => 'Privacy Policy not found.',
                ], 404);
            }

            // Update text if provided
            if ($request->has('text')) {
                $policy->text = $request->text;
            }

            $policy->save();

            return response()->json([
                'status' => true,
                'message' => 'Privacy Policy updated successfully!',
                'data' => [
                    'id' => $policy->id,
                    'text' => $policy->text ?? '',
                ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating the refund policy: ' . $e->getMessage(),
            ], 500);
        }
    }


    public function termsConditions(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        try {
            // Always update ID = 1
            $policy = TermsCondition::find(1);

            if (!$policy) {
                return response()->json([
                    'status' => false,
                    'message' => 'Terms Condition not found.',
                ], 404);
            }

            // Update text if provided
            if ($request->has('text')) {
                $policy->text = $request->text;
            }

            $policy->save();

            return response()->json([
                'status' => true,
                'message' => 'Terms Condition updated successfully!',
                'data' => [
                    'id' => $policy->id,
                    'text' => $policy->text ?? '',
                ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating the refund policy: ' . $e->getMessage(),
            ], 500);
        }
    }


    public function contactUs(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            // Always update ID = 1
            $policy = ContactUs::find(1);

            if (!$policy) {
                return response()->json([
                    'status' => false,
                    'message' => 'Contact Us not found.',
                ], 404);
            }

            // Update text if provided
            if ($request->has('text')) {
                $policy->text = $request->text;
            }

            $policy->save();

            return response()->json([
                'status' => true,
                'message' => 'Contact Us updated successfully!',
                'data' => [
                    'id' => $policy->id,
                    'text' => $policy->text ?? '',
                ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating the refund policy: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function nearbyDistance(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            // Always update ID = 1
            $policy = NearbyDistance::find(1);

            if (!$policy) {
                return response()->json([
                    'status' => false,
                    'message' => 'Nearby Distance not found.',
                ], 404);
            }

            // Update text if provided
            if ($request->has('distance')) {
                $policy->distance = $request->distance;
            }

            if ($request->has('distance_type')) {
                $policy->distance_type = $request->distance_type;
            }

            $policy->save();

            return response()->json([
                'status' => true,
                'message' => 'Nearby Distance updated successfully!',
                'data' => [
                    'id' => $policy->id,
                    'distance' => $policy->distance ?? '',
                    'distance_type' => $policy->distance_type ?? '',
                ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating the refund policy: ' . $e->getMessage(),
            ], 500);
        }
    }



    // addFaq
    public function addFaq(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $rules = [
            'question' => 'required',
            'answer' => 'required',
        ];

        $customMessages = [
            'question.required' => 'Please enter the editCategory.',
            'answer.required' => 'Please enter the answer.',
        ];

        $validator = Validator::make($request->all(), $rules, $customMessages);

        if ($validator->fails()) {
            return response([
                'status' => false,
                'message' => $validator->errors()->first(),
            ], 422);
        }

        $category = new Faq();
        $category->question = $request->question;
        $category->answer = $request->answer;
        $category->save();

        return response([
            'status' => true,
            'message' => 'Faq added successfully!',
        ], 200);
    }




    // editFaq
    public function editFaq(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        try {
            $faq_id = $request->id;

            if (!$faq_id) {
                return response([
                    'status' => false,
                    'message' => 'Faq ID is required.',
                ], 400);
            }

            $category = Faq::find($faq_id);

            if (!$category) {
                return response([
                    'status' => false,
                    'message' => 'Faq not found.',
                ], 404);
            }

            // Update category name if provided
            if ($request->has('question')) {
                $category->question = $request->question;
            }

            if ($request->has('answer')) {
                $category->answer = $request->answer;
            }

            $category->save();

            return response([
                'status' => true,
                'message' => 'Faq updated successfully!',
                'faq' => [
                    'id' => $category->id,
                    'question' => $category->question ?? '',
                    'answer' => $category->answer ?? '',
                ],
            ], 200);
        } catch (\Exception $e) {
            return response([
                'status' => false,
                'message' => 'An error occurred while updating the category: ' . $e->getMessage(),
            ], 500);
        }
    }


    // deleteFaq
    public function deleteFaq(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $request->validate([
            'faq_id' => 'required|exists:faq,id',
        ]);

        $faq_id = $request->faq_id;

        // Find the category
        $category = Faq::find($faq_id);

        if (!$category) {
            return response()->json([
                'status' => false,
                'message' => 'Faq not found.',
            ], 404);
        }

        $category->delete();

        return response()->json([
            'status' => true,
            'message' => 'Faq deleted successfully.',
        ], 200);
    }



    // addReportText
    public function addReportText(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        $rules = [
            'text' => 'required',
        ];

        $customMessages = [
            'text.required' => 'Please enter the text.',
        ];

        $validator = Validator::make($request->all(), $rules, $customMessages);

        if ($validator->fails()) {
            return response([
                'status' => false,
                'message' => $validator->errors()->first(),
            ], 422);
        }

        $category = new ReportText();
        $category->text = $request->text;
        $category->save();

        return response([
            'status' => true,
            'message' => 'Report Text added successfully!',
        ], 200);
    }




    // deleteReportText
    public function deleteReportText(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        $request->validate([
            'report_id' => 'required|exists:report_text,id',
        ]);

        $report_id = $request->report_id;

        // Find the category
        $category = ReportText::find($report_id);

        if (!$category) {
            return response()->json([
                'status' => false,
                'message' => 'Report Text not found.',
            ], 404);
        }

        $category->delete();

        return response()->json([
            'status' => true,
            'message' => 'Report Text deleted successfully.',
        ], 200);
    }



    // clientReviews
    public function clientReviewsold(Request $request)
    {
        $customers = ClientReviews::select('id', 'review_text', 'review_star', 'full_name', 'created_at', 'client_role', 'image')
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($user) {
                return [
                    'id' => $user->id,
                    'review_text' => $user->review_text ?: '',
                    'review_star' => $user->review_star ?: '',
                    'full_name' => $user->full_name ?: '',
                    'created_at' => $user->created_at
                        ? Carbon::parse($user->created_at)->format('d M, Y / g:i A')
                        : '',
                    'client_role' => $user->client_role ?: '',
                    'image' => $user->image
                        ? url('assets/images/index/' . $user->image)
                        : "",
                ];
            });

        return response()->json([
            'status' => true,
            'message' => 'Client Reviews fetched successfully',
            'clientreviews' => $customers
        ]);
    }

    public function clientReviews(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        $search = $request->input('search');
        $page = (int) $request->input('page', 1);
        $pageSize = (int) $request->input('pageSize', 10);
        $orderColumn = $request->input('orderColumn', 'created_at');
        $orderDir = strtoupper($request->input('orderDir', 'DESC'));

        $allowedOrderColumns = ['created_at', 'review_text']; // define allowed columns

        // Validate orderColumn and orderDir
        if (!in_array($orderColumn, $allowedOrderColumns)) {
            $orderColumn = 'created_at';
        }

        if (!in_array($orderDir, ['ASC', 'DESC'])) {
            $orderDir = 'DESC';
        }

        $query = ClientReviews::select('id', 'review_text', 'review_star', 'full_name', 'created_at', 'client_role', 'image');

        if (!empty($search)) {
            $query->where('review_text', 'like', '%' . $search . '%');
        }

        $total = $query->count();
        $data = $query->orderBy($orderColumn, $orderDir)
            ->skip(($page - 1) * $pageSize)
            ->take($pageSize)
            ->get()
            ->map(function ($user) {
                return [
                    'id' => $user->id,
                    'review_text' => $user->review_text ?: '',
                    'review_star' => $user->review_star ?: '',
                    'full_name' => $user->full_name ?: '',
                    'created_at' => $user->created_at
                        ? \Carbon\Carbon::parse($user->created_at)->format('d M, Y / g:i A')
                        : '',
                    'client_role' => $user->client_role ?: '',
                    'image' => $user->image
                        ? url('assets/images/index/' . $user->image)
                        : "",
                ];
            });

        if ($data->isNotEmpty()) {
            return response()->json([
                'status' => true,
                'message' => 'Client Reviews fetched successfully',
                'data' => $data,
                'Pagination' => [
                    'total_pages' => ceil($total / $pageSize),
                    'total_records' => $total,
                    'current_page' => $page,
                    'records_per_page' => $pageSize,
                ]
            ]);
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Client Reviews not found',
            ]);
        }
    }



    // addClientReviews
    public function addClientReviews(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        $rules = [
            'review_text' => 'required',
            'review_star' => 'required',
            'full_name' => 'required',
            'client_role' => 'required',
            'image' => 'required',
        ];

        $customMessages = [
            'review_text.required' => 'Please enter the review text.',
            'review_star.required' => 'Please select a star rating.',
            'full_name.required' => 'Please enter your full name.',
            'client_role.required' => 'Please enter your role or title.',
            'image.required' => 'Please upload an image.',
        ];

        $validator = Validator::make($request->all(), $rules, $customMessages);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'message' => $validator->errors()->first(),
            ], 422);
        }

        // Handle image upload
        $imageName = null;
        if ($request->hasFile('image')) {
            $originalName = pathinfo($request->file('image')->getClientOriginalName(), PATHINFO_FILENAME);
            $sanitizedBase = Str::slug($originalName);
            $extension = $request->file('image')->getClientOriginalExtension();
            $imageName = time() . "_{$sanitizedBase}.{$extension}";
            $request->file('image')->move(public_path('assets/images/index'), $imageName);
        }

        // Create the review
        $review = ClientReviews::create([
            'review_text' => $request->input('review_text'),
            'review_star' => $request->input('review_star'),
            'full_name' => $request->input('full_name'),
            'client_role' => $request->input('client_role'),
            'image' => $imageName,
        ]);

        return response()->json([
            'status' => true,
            'message' => 'Client review added successfully!',
            'data' => $review,
        ], 201);
    }



    // deleteClientReviews
    public function deleteClientReviews(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $request->validate([
            'client_review' => 'required|exists:client_reviews,id',
        ]);

        $client_review = $request->client_review;

        // Find the category
        $category = ClientReviews::find($client_review);

        if (!$category) {
            return response()->json([
                'status' => false,
                'message' => 'Client Review not found.',
            ], 404);
        }

        $category->delete();

        return response()->json([
            'status' => true,
            'message' => 'Client Review deleted successfully.',
        ], 200);
    }

    public function deleteCustomerSupport(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        $request->validate([
            'customersupport_id' => 'required|exists:customer_support,id',
        ]);

        $client_review = $request->customersupport_id;

        // Find the category
        $category = CustomerSupport::find($client_review);

        if (!$category) {
            return response()->json([
                'status' => false,
                'message' => 'Customer Support not found.',
            ], 404);
        }

        $category->delete();

        return response()->json([
            'status' => true,
            'message' => 'Customer Support deleted successfully.',
        ], 200);
    }



    // paymentUpdate
    public function paymentUpdate(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            $id = $request->id;

            if (!$id) {
                return response()->json([
                    'status' => false,
                    'message' => 'Payment ID is required.',
                ], 400);
            }

            $paymentGateway = PaymentGatewayKey::find($id);

            if (!$paymentGateway) {
                return response()->json([
                    'status' => false,
                    'message' => 'Payment gateway not found.',
                ], 404);
            }

            // Update fields if provided
            $fieldsToUpdate = [
                'public_key',
                'secret_key',
                'mode',
                'status',
                'country_code',
                'currency_code',
            ];

            foreach ($fieldsToUpdate as $field) {
                if ($request->has($field)) {
                    $paymentGateway->$field = $request->$field;
                }
            }

            $paymentGateway->save();

            return response()->json([
                'status' => true,
                'message' => 'Payment gateway updated successfully!',
                'data' => [
                    'id' => $paymentGateway->id,
                    'text' => $paymentGateway->text ?? '',
                    'public_key' => $paymentGateway->public_key ?? '',
                    'secret_key' => $paymentGateway->secret_key ?? '',
                    'mode' => $paymentGateway->mode ?? '',
                    'status' => $paymentGateway->status,
                    'country_code' => $paymentGateway->country_code ?? '',
                    'currency_code' => $paymentGateway->currency_code ?? '',
                ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating the payment gateway: ' . $e->getMessage(),
            ], 500);
        }
    }


    // feedbackList
    public function feedbackListold(Request $request)
    {
        $feedbacks = AppFeedback::with('user:id,first_name,last_name,email,image') // eager load user data
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($feedback) {
                $user = $feedback->user;

                return [
                    'id' => $feedback->id,
                    'feedback_review' => $feedback->feedback_review ?: '',
                    'created_at' => $feedback->created_at
                        ? Carbon::parse($feedback->created_at)->format('d M, Y / g:i A')
                        : '',
                    'first_name' => $user->first_name ?? '',
                    'last_name' => $user->last_name ?? '',
                    'email' => $user->email ?? '',
                    'image' => isset($user->image) && $user->image
                        ? url('assets/images/users_images/' . $user->image)
                        : "",
                    'status' => $feedback->status ?? 1, // defaulting to 1 if not present
                ];
            });

        return response()->json([
            'status' => true,
            'message' => 'App feedback list fetched successfully.',
            'customers' => $feedbacks,
        ]);
    }

    public function feedbackListold2(Request $request)
    {
        $page = max((int) $request->input('page', 1), 1);
        $pageSize = max((int) $request->input('pageSize', 10), 1);
        $orderColumn = $request->input('orderColumn', 'created_at');
        $orderDir = strtoupper($request->input('orderDir', 'DESC'));

        $allowedOrderColumns = ['created_at', 'feedback_review'];
        if (!in_array($orderColumn, $allowedOrderColumns)) {
            $orderColumn = 'created_at';
        }

        if (!in_array($orderDir, ['ASC', 'DESC'])) {
            $orderDir = 'DESC';
        }

        $query = AppFeedback::with('user:id,first_name,last_name,email,image')
            ->orderBy($orderColumn, $orderDir);

        $total = $query->count();

        $feedbacks = $query->skip(($page - 1) * $pageSize)
            ->take($pageSize)
            ->get()
            ->map(function ($feedback) {
                $user = $feedback->user;

                return [
                    'id' => $feedback->id,
                    'feedback_review' => $feedback->feedback_review ?: '',
                    'created_at' => $feedback->created_at
                        ?? '',
                    'first_name' => $user->first_name ?? '',
                    'last_name' => $user->last_name ?? '',
                    'email' => $user->email ?? '',
                    'image' => isset($user->image) && $user->image
                        ? url('assets/images/users_images/' . $user->image)
                        : "",
                    'status' => $feedback->status ?? 1,
                ];
            });

        return response()->json([
            'status' => true,
            'message' => 'App feedback list fetched successfully.',
            'customers' => $feedbacks,
            'Pagination' => [
                'total_pages' => ceil($total / $pageSize),
                'total_records' => $total,
                'current_page' => $page,
                'records_per_page' => $pageSize,
            ]
        ]);
    }

    public function feedbackList(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        $page = max((int) $request->input('page', 1), 1);
        $pageSize = max((int) $request->input('pageSize', 10), 1);
        $orderColumn = $request->input('orderColumn', 'created_at');
        $orderDir = strtoupper($request->input('orderDir', 'DESC'));
        $search = $request->input('search');

        $allowedOrderColumns = ['created_at', 'feedback_review'];
        if (!in_array($orderColumn, $allowedOrderColumns)) {
            $orderColumn = 'created_at';
        }

        if (!in_array($orderDir, ['ASC', 'DESC'])) {
            $orderDir = 'DESC';
        }

        $query = AppFeedback::with('user:id,first_name,last_name,email,image')
            ->when($search, function ($q) use ($search) {
                $q->whereHas('user', function ($q2) use ($search) {
                    $q2->where('first_name', 'like', "%$search%")
                        ->orWhere('last_name', 'like', "%$search%")
                        ->orWhere('email', 'like', "%$search%");
                });
            })
            ->orderBy($orderColumn, $orderDir);

        $total = $query->count();

        $feedbacks = $query->skip(($page - 1) * $pageSize)
            ->take($pageSize)
            ->get()
            ->map(function ($feedback) {
                $user = $feedback->user;

                return [
                    'id' => $feedback->id,
                    'feedback_review' => $feedback->feedback_review ?: '',
                    'created_at' => $feedback->created_at ?? '',
                    'first_name' => $user->first_name ?? '',
                    'last_name' => $user->last_name ?? '',
                    'email' => $user->email ?? '',
                    'image' => isset($user->image) && $user->image
                        ? url('assets/images/users_images/' . $user->image)
                        : "",
                    'status' => $feedback->status ?? 1,
                ];
            });

        return response()->json([
            'status' => true,
            'message' => 'App feedback list fetched successfully.',
            'customers' => $feedbacks,
            'Pagination' => [
                'total_pages' => ceil($total / $pageSize),
                'total_records' => $total,
                'current_page' => $page,
                'records_per_page' => $pageSize,
            ]
        ]);
    }





    public function customerSupportold(Request $request)
    {
        $feedbacks = CustomerSupport::orderBy('created_at', 'desc')
            ->get()
            ->map(function ($feedback) {
                return [
                    'id' => $feedback->id,
                    'name' => $feedback->name ?? '',
                    'phone' => $feedback->phone ?? '',
                    'email' => $feedback->email ?? '',
                    'message' => $feedback->message ?? '',
                    'created_at' => $feedback->created_at
                        ? Carbon::parse($feedback->created_at)->format('d M, Y / g:i A')
                        : '',
                ];
            });

        return response()->json([
            'status' => true,
            'message' => 'Customer support list fetched successfully.',
            'customers' => $feedbacks,
        ]);
    }

    public function customerSupport(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        $page = max((int) $request->input('page', 1), 1);
        $pageSize = max((int) $request->input('pageSize', 10), 1);
        $orderColumn = $request->input('orderColumn', 'created_at');
        $orderDir = strtoupper($request->input('orderDir', 'DESC'));

        $allowedOrderColumns = ['created_at', 'name', 'email'];
        if (!in_array($orderColumn, $allowedOrderColumns)) {
            $orderColumn = 'created_at';
        }

        if (!in_array($orderDir, ['ASC', 'DESC'])) {
            $orderDir = 'DESC';
        }

        $search = $request->input('search'); // name or email

        $query = CustomerSupport::query();

        if (!empty($search)) {
            $query->where(function ($q) use ($search) {
                $q->where('name', 'LIKE', "%$search%")
                    ->orWhere('email', 'LIKE', "%$search%");
            });
        }

        $total = $query->count();

        $feedbacks = $query->orderBy($orderColumn, $orderDir)
            ->skip(($page - 1) * $pageSize)
            ->take($pageSize)
            ->get()
            ->map(function ($feedback) {
                return [
                    'id' => $feedback->id,
                    'name' => $feedback->name ?? '',
                    'phone' => $feedback->phone ?? '',
                    'email' => $feedback->email ?? '',
                    'message' => $feedback->message ?? '',
                    'created_at' => $feedback->created_at ?? '',
                ];
            });

        return response()->json([
            'status' => true,
            'message' => 'Customer support list fetched successfully.',
            'customers' => $feedbacks,
            'Pagination' => [
                'total_pages' => ceil($total / $pageSize),
                'total_records' => $total,
                'current_page' => $page,
                'records_per_page' => $pageSize,
            ]
        ]);
    }





    // subscribedEmail
    public function subscribedEmailold(Request $request)
    {
        $feedbacks = NewsEmail::orderBy('created_at', 'desc')
            ->get()
            ->map(function ($feedback) {
                return [
                    'id' => $feedback->id,
                    'email' => $feedback->email ?? '',
                    'created_at' => $feedback->created_at
                        ? Carbon::parse($feedback->created_at)->format('d M, Y / g:i A')
                        : '',
                ];
            });

        return response()->json([
            'status' => true,
            'message' => 'Subscribed email list fetched successfully.',
            'customers' => $feedbacks,
        ]);
    }

    public function subscribedEmail(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $page = max((int) $request->input('page', 1), 1);
        $pageSize = max((int) $request->input('pageSize', 10), 1);
        $orderColumn = $request->input('orderColumn', 'created_at');
        $orderDir = strtoupper($request->input('orderDir', 'DESC'));

        $allowedOrderColumns = ['created_at', 'email'];
        if (!in_array($orderColumn, $allowedOrderColumns)) {
            $orderColumn = 'created_at';
        }

        if (!in_array($orderDir, ['ASC', 'DESC'])) {
            $orderDir = 'DESC';
        }

        $search = $request->input('search'); // Search email

        $query = NewsEmail::query();

        if (!empty($search)) {
            $query->where('email', 'LIKE', "%$search%");
        }

        $total = $query->count();

        $feedbacks = $query->orderBy($orderColumn, $orderDir)
            ->skip(($page - 1) * $pageSize)
            ->take($pageSize)
            ->get()
            ->map(function ($feedback) {
                return [
                    'id' => $feedback->id,
                    'email' => $feedback->email ?? '',
                    'created_at' => $feedback->created_at ?? '',
                ];
            });

        return response()->json([
            'status' => true,
            'message' => 'Subscribed email list fetched successfully.',
            'customers' => $feedbacks,
            'Pagination' => [
                'total_pages' => ceil($total / $pageSize),
                'total_records' => $total,
                'current_page' => $page,
                'records_per_page' => $pageSize,
            ]
        ]);
    }




    public function generalSettings(Request $request)
    {
        try {
            // Always update ID = 1
            $siteSettings = SiteSetup::find(1);

            if (!$siteSettings) {
                return response()->json([
                    'status' => false,
                    'message' => 'General settings not found.',
                ], 404);
            }

            // Fields to update if provided
            $fields = [
                'name',
                'default_currency',
                'default_currency_name',
                'color_code',
                'copyright_text',
                'email',
                'purchase_code',
            ];

            foreach ($fields as $field) {
                if ($request->has($field)) {
                    $siteSettings->$field = $request->$field;
                }
            }

            // Image fields
            $imageFields = [
                'light_logo_url' => 'light_logo',
                'dark_logo_url' => 'dark_logo',
                'fav_icon_url' => 'fav_icon',
            ];

            foreach ($imageFields as $inputName => $columnName) {
                if ($request->hasFile($inputName)) {
                    $file = $request->file($inputName);
                    $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                    $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                    $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                    $file->move(public_path('assets/images/logo'), $fileName);

                    $siteSettings->$columnName = $fileName;
                }
            }

            $siteSettings->save();


            $done = MobileUrl::first();

            return response()->json([
                'status' => true,
                'message' => 'General settings updated successfully!',
                'data' => [
                    'id' => $siteSettings->id,
                    'name' => $siteSettings->name ?? '',
                    'default_currency' => $siteSettings->default_currency ?? '',
                    'default_currency_name' => $siteSettings->default_currency_name ?? '',
                    'color_code' => $siteSettings->color_code ?? '',
                    'copyright_text' => $siteSettings->copyright_text ?? '',
                    'light_logo_url' => $siteSettings->light_logo
                        ? asset('assets/images/logo/' . $siteSettings->light_logo)
                        : '',
                    'dark_logo_url' => $siteSettings->dark_logo
                        ? asset('assets/images/logo/' . $siteSettings->dark_logo)
                        : '',
                    'fav_icon_url' => $siteSettings->fav_icon
                        ? asset('assets/images/logo/' . $siteSettings->fav_icon)
                        : '',
                    'email' => $siteSettings->email ?? '',
                    'android_url' => $done->android_url ?? "",
                    'ios_url' => $done->ios_url ?? "",
                    'purchase_code' => $siteSettings->purchase_code ?? "",
                    'is_payment' => $siteSettings->is_extended ? ($siteSettings->is_extended = 1 ? true : false) : false,

                ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating general settings: ' . $e->getMessage(),
            ], 500);
        }
    }




    public function mobileSetup(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            // Always update ID = 1
            $siteSettings = MobileUrl::find(1);

            if (!$siteSettings) {
                return response()->json([
                    'status' => false,
                    'message' => 'Mobile setup settings not found.',
                ], 404);
            }

            // Fields to update if provided
            $fields = [
                'android_url',
                'facebook_link',
                'ios_url',
                'ios_provider_url',
                'android_provider_url',
            ];

            foreach ($fields as $field) {
                if ($request->filled($field)) {
                    $siteSettings->$field = $request->$field;
                }
            }

            $siteSettings->save();

            return response()->json([
                'status' => true,
                'message' => 'Mobile setup settings updated successfully!',
                'data' => [
                    'id' => $siteSettings->id,
                    'android_url' => $siteSettings->android_url ?? '',
                    'android_provider_url' => $siteSettings->android_provider_url ?? '',
                    'ios_url' => $siteSettings->ios_url ?? '',
                    'ios_provider_url' => $siteSettings->ios_provider_url ?? '',
                ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating mobile setup settings: ' . $e->getMessage(),
            ], 500);
        }
    }



    public function socialMedia(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            // Always update ID = 1
            $siteSettings = SocialMedia::find(1);

            if (!$siteSettings) {
                return response()->json([
                    'status' => false,
                    'message' => 'Social media settings not found.',
                ], 404);
            }

            // Fields to update if provided
            $fields = [
                'facebook_link',
                'whatsapp_link',
                'instagram_link',
                'twitter_link',
                'youtube_link',
                'linkdln_link',
                'share_link'
            ];

            foreach ($fields as $field) {
                if ($request->filled($field)) {
                    $siteSettings->$field = $request->$field;
                }
            }

            $siteSettings->save();

            return response()->json([
                'status' => true,
                'message' => 'Social media settings updated successfully!',
                'data' => [
                    'id' => $siteSettings->id,
                    'facebook_link' => $siteSettings->facebook_link ?? '',
                    'whatsapp_link' => $siteSettings->whatsapp_link ?? '',
                    'instagram_link' => $siteSettings->instagram_link ?? '',
                    'twitter_link' => $siteSettings->twitter_link ?? '',
                    'youtube_link' => $siteSettings->youtube_link ?? '',
                    'linkdln_link' => $siteSettings->linkdln_link ?? '',
                    'share_link' => $siteSettings->share_link ?? '',
                ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating social media settings: ' . $e->getMessage(),
            ], 500);
        }
    }





    public function mailSetup(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            // Always update ID = 1
            $siteSettings = MailSetup::find(1);

            if (!$siteSettings) {
                return response()->json([
                    'status' => false,
                    'message' => 'Mail setup settings not found.',
                ], 404);
            }

            // Fields to update if provided
            $fields = [
                'mail_mailer',
                'mail_host',
                'mail_port',
                'mail_encryption',
                'mail_username',
                'mail_password',
                'mail_from',
            ];

            foreach ($fields as $field) {
                if ($request->filled($field)) {
                    $siteSettings->$field = $request->$field;
                }
            }

            $siteSettings->save();

            return response()->json([
                'status' => true,
                'message' => 'Mail setup settings updated successfully!',
                'data' => [
                    'id' => $siteSettings->id,
                    'mail_mailer' => $siteSettings->mail_mailer ?? '',
                    'mail_host' => $siteSettings->mail_host ?? '',
                    'mail_port' => $siteSettings->mail_port ?? '',
                    'mail_encryption' => $siteSettings->mail_encryption ?? '',
                    'mail_username' => $siteSettings->mail_username ?? '',
                    'mail_password' => $siteSettings->mail_password ?? '',
                    'mail_from' => $siteSettings->mail_from ?? '',
                ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating mail setup settings: ' . $e->getMessage(),
            ], 500);
        }
    }






    public function indexSliderUpdate(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            // Always update ID = 1
            $slider = IndexSlider::find(1);

            if (!$slider) {
                return response()->json([
                    'status' => false,
                    'message' => 'Index slider not found.',
                ], 404);
            }

            // Update fields if provided
            $fieldsToUpdate = [
                'title',
                'body',
                'link',
            ];

            foreach ($fieldsToUpdate as $field) {
                if ($request->filled($field)) {
                    $slider->$field = $request->$field;
                }
            }

            // Handle image upload if present
            if ($request->hasFile('image')) {
                $file = $request->file('image');
                $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                $file->move(public_path('assets/images/index'), $fileName);

                // Update the image field
                $slider->image = $fileName;
            }

            $slider->save();

            $status_all = HomeSettings::where('id', "1")->first();
            $done = $status_all->status;

            return response()->json([
                'status' => true,
                'message' => 'Index slider updated successfully!',
                'data' => [
                    'id' => $slider->id,
                    'title' => $slider->title ?? '',
                    'body' => $slider->body ?? '',
                    'link' => $slider->link ?? '',
                    'main_id' => 1,
                    'status' => $done ?? 0,
                    // 'image' => $slider->image ?? '',
                    'image' => $slider->image
                        ? asset('assets/images/index/' . $slider->image)
                        : '',
                ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating the index slider: ' . $e->getMessage(),
            ], 500);
        }
    }




    public function indexCategoryUpdateold(Request $request)
    {
        try {
            // Validate the request
            $request->validate([
                'category_id' => 'required|string', // expects comma-separated string like "1,2,3"
            ]);

            // Truncate the table before inserting new data
            IndexCategories::truncate();

            // Split the input string into an array
            $categoryIds = explode(',', $request->category_id);

            // Insert each category_id as a new row
            foreach ($categoryIds as $categoryId) {
                IndexCategories::create([
                    'category_id' => trim($categoryId),
                ]);
            }

            return response()->json([
                'status' => true,
                'message' => 'Index categories updated successfully!',
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating index categories: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function indexCategoryUpdate2(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            // Validate the request
            // $request->validate([
            //     'category_id' => 'required|string', // expects comma-separated string like "1,2,3"
            // ]);

            // Truncate the table before inserting new data
            IndexCategories::truncate();

            // Split the input string into an array and trim each element
            $categoryIds = array_filter(array_map('trim', explode(',', $request->category_id)));

            $insertedCategories = [];

            foreach ($categoryIds as $categoryId) {
                // Optional: Check if the category exists
                $category = Category::find($categoryId);
                if ($category) {
                    IndexCategories::create([
                        'category_id' => $categoryId,
                    ]);

                    // Add category name to response list
                    $insertedCategories[] = [
                        'id' => $category->id,
                        'name' => $category->category_name, // Adjust if your column name differs
                    ];
                }
            }

            return response()->json([
                'status' => true,
                'message' => 'Index categories updated successfully!',
                'categories' => $insertedCategories, // return category names
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating index categories: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function indexCategoryUpdate(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            $categoryIds = [];

            // Only update if category_id is provided and not empty
            if ($request->has('category_id') && trim($request->category_id) !== '') {
                // Truncate the table before inserting new data
                IndexCategories::truncate();

                // Parse comma-separated category IDs
                $categoryIds = array_filter(array_map('trim', explode(',', $request->category_id)));

                foreach ($categoryIds as $categoryId) {
                    $category = Category::find($categoryId);
                    if ($category) {
                        IndexCategories::create([
                            'category_id' => $categoryId,
                        ]);
                    }
                }
            }

            $status_all = HomeSettings::where('id', "2")->first();
            $done = $status_all->status;



            // // Now fetch the current records from `index_categories`
            // $currentCategories = IndexCategories::with('category:id,category_name') // Assuming relation `category()`
            //     ->get()
            //     ->map(function ($item) {
            //         return [
            //             'id' => $item->category_id,
            //             'name' => optional($item->category)->category_name,
            //             'main_id' => 2,
            //             'status' => $done,

            //         ];
            //     });

            $currentCategories = IndexCategories::with('category:id,category_name') // Assuming relation `category()`
                ->get()
                ->map(function ($item) use ($done) {
                    return [
                        'id' => $item->category_id,
                        'name' => optional($item->category)->category_name,
                        'main_id' => 2,
                        'status' => $done,
                    ];
                });

            return response()->json([
                'status' => true,
                'message' => $request->has('category_id') && trim($request->category_id) !== ''
                    ? 'Index categories updated successfully!'
                    : 'No category_id provided; returning existing index categories.',
                'categories' => $currentCategories,
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred: ' . $e->getMessage(),
            ], 500);
        }
    }



    public function indexCard1Update(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            // Always update the card with ID = 1
            $card = IndexCards::find(1);

            if (!$card) {
                return response()->json([
                    'status' => false,
                    'message' => 'Index card not found.',
                ], 404);
            }

            // Fields that can be updated
            $fieldsToUpdate = [
                'category_id',
                'title',
                'image',
            ];

            // Update the fields if provided
            foreach ($fieldsToUpdate as $field) {
                if ($request->filled($field)) {
                    $card->$field = $request->$field;
                }
            }

            // Handle image upload if present
            if ($request->hasFile('image')) {
                $file = $request->file('image');
                $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                $file->move(public_path('assets/images/index'), $fileName);

                // Update image field with new file name
                $card->image = $fileName;
            }

            // Save the updated card
            $card->save();



            $status_all = HomeSettings::where('id', "3")->first();
            $done = $status_all->status;



            return response()->json([
                'status' => true,
                'message' => 'Index card updated successfully!',
                'data' => [
                    'id' => $card->id,
                    'title' => $card->title ?? '',
                    'image' => $card->image ? asset('assets/images/index/' . $card->image) : '',
                    'category_id' => $card->category_id ?? '',
                    'main_id' => 3,
                    'status' => $done ?? 0,
                ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating the index card: ' . $e->getMessage(),
            ], 500);
        }
    }


    public function indexCard2Update(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            // Always update the card with ID = 1
            $card = IndexCards::find(2);

            if (!$card) {
                return response()->json([
                    'status' => false,
                    'message' => 'Index card not found.',
                ], 404);
            }

            // Fields that can be updated
            $fieldsToUpdate = [
                'category_id',
                'title',
                'image',
            ];

            // Update the fields if provided
            foreach ($fieldsToUpdate as $field) {
                if ($request->filled($field)) {
                    $card->$field = $request->$field;
                }
            }

            // Handle image upload if present
            if ($request->hasFile('image')) {
                $file = $request->file('image');
                $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                $file->move(public_path('assets/images/index'), $fileName);

                // Update image field with new file name
                $card->image = $fileName;
            }

            // Save the updated card
            $card->save();

            $status_all = HomeSettings::where('id', "3")->first();
            $done = $status_all->status;

            return response()->json([
                'status' => true,
                'message' => 'Index card updated successfully!',
                'data' => [
                    'id' => $card->id,
                    'title' => $card->title ?? '',
                    // 'image' => $card->image ?? '',
                    'image' => $card->image ? asset('assets/images/index/' . $card->image) : '',
                    'category_id' => $card->category_id ?? '',
                    'main_id' => 3,
                    'status' => $done ?? 0,
                ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating the index card: ' . $e->getMessage(),
            ], 500);
        }
    }



    public function indexCard3Update(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            // Always update the card with ID = 1
            $card = IndexCards::find(3);

            if (!$card) {
                return response()->json([
                    'status' => false,
                    'message' => 'Index card not found.',
                ], 404);
            }

            // Fields that can be updated
            $fieldsToUpdate = [
                'category_id',
                'title',
                'image',
            ];

            // Update the fields if provided
            foreach ($fieldsToUpdate as $field) {
                if ($request->filled($field)) {
                    $card->$field = $request->$field;
                }
            }

            // Handle image upload if present
            if ($request->hasFile('image')) {
                $file = $request->file('image');
                $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                $file->move(public_path('assets/images/index'), $fileName);

                // Update image field with new file name
                $card->image = $fileName;
            }

            // Save the updated card
            $card->save();

            $status_all = HomeSettings::where('id', "3")->first();
            $done = $status_all->status;

            return response()->json([
                'status' => true,
                'message' => 'Index card updated successfully!',
                'data' => [
                    'id' => $card->id,
                    'title' => $card->title ?? '',
                    'image' => $card->image ? asset('assets/images/index/' . $card->image) : '',
                    'category_id' => $card->category_id ?? '',
                    'main_id' => 3,
                    'status' => $done ?? 0,
                ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating the index card: ' . $e->getMessage(),
            ], 500);
        }
    }



    public function indexCard4Update(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            // Always update the card with ID = 1
            $card = IndexCards::find(4);

            if (!$card) {
                return response()->json([
                    'status' => false,
                    'message' => 'Index card not found.',
                ], 404);
            }

            // Fields that can be updated
            $fieldsToUpdate = [
                'category_id',
                'title',
                'image',
            ];

            // Update the fields if provided
            foreach ($fieldsToUpdate as $field) {
                if ($request->filled($field)) {
                    $card->$field = $request->$field;
                }
            }

            // Handle image upload if present
            if ($request->hasFile('image')) {
                $file = $request->file('image');
                $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                $file->move(public_path('assets/images/index'), $fileName);

                // Update image field with new file name
                $card->image = $fileName;
            }

            // Save the updated card
            $card->save();

            $status_all = HomeSettings::where('id', "3")->first();
            $done = $status_all->status;

            return response()->json([
                'status' => true,
                'message' => 'Index card updated successfully!',
                'data' => [
                    'id' => $card->id,
                    'title' => $card->title ?? '',
                    'image' => $card->image ? asset('assets/images/index/' . $card->image) : '',
                    'category_id' => $card->category_id ?? '',
                    'main_id' => 3,
                    'status' => $done ?? 0,
                ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating the index card: ' . $e->getMessage(),
            ], 500);
        }
    }


    // indexCitiesUpdate
    public function indexCitiesUpdateold(Request $request)
    {
        try {
            // Validate the request
            $request->validate([
                'service_id' => 'required|string',
                'title' => 'nullable|string',
                'subtext' => 'nullable|string',
            ]);

            // Truncate the table before inserting new data
            IndexCities::truncate();

            // Split the input string into an array
            $serviceIds = explode(',', $request->service_id);

            // Insert each service_id as a new row with shared title and subtext
            foreach ($serviceIds as $id) {
                IndexCities::create([
                    'service_id' => trim($id),
                    'title' => $request->title ?? null,
                    'subtext' => $request->subtext ?? null,
                ]);
            }

            return response()->json([
                'status' => true,
                'message' => 'Index cities updated successfully!',
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating index cities: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function indexCitiesUpdate(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            $serviceIds = [];

            // Only update if service_id is provided and not empty
            if ($request->has('city') && trim($request->city) !== '') {
                IndexCities::truncate();

                $cities = array_filter(array_map('trim', explode(',', $request->city)));

                foreach ($cities as $cityName) {
                    IndexCities::create([
                        'city' => $cityName,
                        'title' => $request->title ?? null,
                        'subtext' => $request->subtext ?? null,
                    ]);
                }
            }

            $status_all = HomeSettings::where('id', "5")->first();
            $done = $status_all->status;

            // Return all current IndexCities with service name
            $currentCities = IndexCities::get()
                ->map(function ($item) use ($done) {
                    return [
                        'title' => $item->title,
                        'subtext' => $item->subtext,
                        'city' => $item->city,
                        'main_id' => 5,
                        'status' => $done,
                    ];
                });

            return response()->json([
                'status' => true,
                'message' => $request->has('city') && trim($request->city) !== ''
                    ? 'Index cities updated successfully!'
                    : 'No service_id provided; returning existing index cities.',
                'cities' => $currentCities,
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function indexSponserStore(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            $serviceIds = [];

            // Only update if service_id is provided and not empty
            if ($request->has('service_id') && trim($request->service_id) !== '') {
                // Truncate table before inserting new data
                IndexSponserStores::truncate();

                // Split and clean service_id list
                $serviceIds = array_filter(array_map('trim', explode(',', $request->service_id)));

                foreach ($serviceIds as $id) {
                    IndexSponserStores::create([
                        'service_id' => $id,
                        'title' => $request->title ?? null,
                        'subtext' => $request->subtext ?? null,
                    ]);
                }
            }


            $status_all = HomeSettings::where('id', "4")->first();
            $done = $status_all->status;


            // Return all current IndexCities with service name
            $currentCities = IndexSponserStores::with('service:id,service_name') // assuming relation exists
                ->get()
                ->map(function ($item) use ($done) {
                    return [
                        'service_id' => $item->service_id,
                        'service_name' => optional($item->service)->service_name,
                        'title' => $item->title,
                        'subtext' => $item->subtext,
                        'main_id' => 4,
                        'status' => $done,
                    ];
                });

            return response()->json([
                'status' => true,
                'message' => $request->has('service_id') && trim($request->service_id) !== ''
                    ? 'Index Sponser Store updated successfully!'
                    : 'No service_id provided; returning existing index cities.',
                'sponser_store' => $currentCities,
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function indexBannerUpdate(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            // Always update ID = 1
            $slider = IndexBanner::find(1);

            if (!$slider) {
                return response()->json([
                    'status' => false,
                    'message' => 'Index Banner not found.',
                ], 404);
            }

            // Update fields if provided
            $fieldsToUpdate = [
                'play_store_url',
                'app_store_url',
            ];

            foreach ($fieldsToUpdate as $field) {
                if ($request->filled($field)) {
                    $slider->$field = $request->$field;
                }
            }

            // Handle image upload if present
            if ($request->hasFile('image')) {
                $file = $request->file('image');
                $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                $file->move(public_path('assets/images/index'), $fileName);

                // Update the image field
                $slider->image = $fileName;
            }

            $slider->save();

            $status_all = HomeSettings::where('id', "6")->first();
            $done = $status_all->status;

            return response()->json([
                'status' => true,
                'message' => 'Index Banner updated successfully!',
                'data' => [
                    'id' => $slider->id,
                    'play_store_url' => $slider->play_store_url ?? '',
                    'app_store_url' => $slider->app_store_url ?? '',
                    'image' => $slider->image
                        ? asset('assets/images/index/' . $slider->image)
                        : '',
                    'main_id' => 6,
                    'status' => $done ?? 0,
                ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating the index slider: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function indexTestimonialsUpdate(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            // Always update ID = 1
            $slider = IndexTestimonials::find(1);

            if (!$slider) {
                return response()->json([
                    'status' => false,
                    'message' => 'Index Testimonials not found.',
                ], 404);
            }

            // Update fields if provided
            $fieldsToUpdate = [
                'title',
                'subtext',
                'body',
            ];

            foreach ($fieldsToUpdate as $field) {
                if ($request->filled($field)) {
                    $slider->$field = $request->$field;
                }
            }

            // Handle image upload if present
            if ($request->hasFile('image')) {
                $file = $request->file('image');
                $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                $file->move(public_path('assets/images/index'), $fileName);

                // Update the image field
                $slider->image = $fileName;
            }

            $slider->save();

            $status_all = HomeSettings::where('id', "8")->first();

            $done = $status_all->status;





            return response()->json([
                'status' => true,
                'message' => 'Index Testimonials updated successfully!',
                'data' => [
                    'id' => $slider->id,
                    'title' => $slider->title ?? '',
                    'subtext' => $slider->subtext ?? '',
                    'body' => $slider->body ?? '',
                    'image' => $slider->image
                        ? asset('assets/images/index/' . $slider->image)
                        : '',
                    'main_id' => 8,
                    'status' => $done ?? 0,
                ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating the index slider: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function indexSeoUpdate(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            // Always update ID = 1
            $slider = HomeSeo::find(1);

            if (!$slider) {
                return response()->json([
                    'status' => false,
                    'message' => 'Index Seo not found.',
                ], 404);
            }

            // Update fields if provided
            $fieldsToUpdate = [
                'title',
                'body',
            ];

            foreach ($fieldsToUpdate as $field) {
                if ($request->filled($field)) {
                    $slider->$field = $request->$field;
                }
            }

            // Handle image upload if present
            if ($request->hasFile('image')) {
                $file = $request->file('image');
                $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                $file->move(public_path('assets/images/index'), $fileName);

                // Update the image field
                $slider->image = $fileName;
            }

            $slider->save();



            $status_all = HomeSettings::where('id', "9")->first();

            $done = $status_all->status;



            return response()->json([
                'status' => true,
                'message' => 'Index Seo updated successfully!',
                'data' => [
                    'id' => $slider->id,
                    'title' => $slider->title ?? '',
                    'body' => $slider->body ?? '',
                    'image' => $slider->image
                        ? asset('assets/images/index/' . $slider->image)
                        : '',
                    'main_id' => 9,
                    'status' => $done ?? 0,
                ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating the index slider: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function indexPerfectStore(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            $serviceIds = [];

            // Only update if service_id is provided and not empty
            if ($request->has('service_id') && trim($request->service_id) !== '') {
                // Truncate table before inserting new data
                IndexPerfectStore::truncate();

                // Split and clean service_id list
                $serviceIds = array_filter(array_map('trim', explode(',', $request->service_id)));

                foreach ($serviceIds as $id) {
                    IndexPerfectStore::create([
                        'service_id' => $id,
                        'title' => $request->title ?? null,
                        'subtext' => $request->subtext ?? null,
                    ]);
                }
            }

            $status_all = HomeSettings::where('id', "7")->first();

            $done = $status_all->status;

            // Return all current IndexCities with service name
            $currentCities = IndexPerfectStore::with('service:id,service_name') // assuming relation exists
                ->get()
                ->map(function ($item) use ($done) {
                    return [
                        'service_id' => $item->service_id,
                        'service_name' => optional($item->service)->service_name,
                        'title' => $item->title,
                        'subtext' => $item->subtext,
                        'main_id' => 7,
                        'status' => $done ?? 0,
                    ];
                });

            return response()->json([
                'status' => true,
                'message' => $request->has('service_id') && trim($request->service_id) !== ''
                    ? 'Index Sponser Store updated successfully!'
                    : 'No service_id provided; returning existing index cities.',
                'sponser_store' => $currentCities,
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred: ' . $e->getMessage(),
            ], 500);
        }
    }


    public function dashboardList(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            // Totals
            $totalServices = Service::where('request_approval', 1)->count();
            $totalstores = Stores::count();
            $totalUsers = User::where('role', 'user')->count();
            $totalVendors = User::where('role', 'vendor')->count();
            $totalSubscriptions = UserSubscription::count();
            $totalPrice = UserSubscriptionHistory::sum('price');
            $totalGoal = GoalPaymentHistory::sum('price');

            $month = Carbon::now()->month; // Optional, if you still want to use it elsewhere

            $user_permonth = User::where('role', 'user')->whereMonth('created_at', Carbon::now()->month)
                ->whereYear('created_at', Carbon::now()->year)
                ->count();

            $lastMonth = Carbon::now()->subMonth(); // Get last month date

            $user_last_month = User::where('role', 'user')->whereMonth('created_at', $lastMonth->month)
                ->whereYear('created_at', $lastMonth->year)
                ->count();

            $sum_of_user = $user_permonth - $user_last_month;

            // $user_persontage = round($sum_of_user * 100 / $totalUsers);

            if ($totalUsers > 0) {
                $user_persontage = round($sum_of_user * 100 / $totalUsers);
            } else {
                $user_persontage = 0; // Or handle as needed
            }


            $store_permonth = Stores::whereMonth('created_at', Carbon::now()->month)
                ->whereYear('created_at', Carbon::now()->year)
                ->count();

            $lastMonth = Carbon::now()->subMonth(); // Get last month date

            $store_last_month = Stores::whereMonth('created_at', $lastMonth->month)
                ->whereYear('created_at', $lastMonth->year)
                ->count();

            $sum_of_store = $store_permonth - $store_last_month;

            // $store_persontage = round($sum_of_store * 100 / $totalstores);


            if ($totalstores > 0) {
                $store_persontage = round($sum_of_store * 100 / $totalstores);
            } else {
                $store_persontage = 0; // Or handle as needed
            }

            $service_permonth = Service::whereMonth('created_at', Carbon::now()->month)
                ->whereYear('created_at', Carbon::now()->year)
                ->count();

            $lastMonth = Carbon::now()->subMonth(); // Get last month date


            $service_last_month = Service::whereMonth('created_at', $lastMonth->month)
                ->whereYear('created_at', $lastMonth->year)
                ->count();

            $sum_of_service = $service_permonth - $service_last_month;

            // $service_persontage = round($sum_of_service * 100 / $totalServices);

            if ($totalServices > 0) {
                $service_persontage = round($sum_of_service * 100 / $totalServices);
            } else {
                $service_persontage = 0; // Or handle as needed
            }

            $subcription_permonth = UserSubscription::whereMonth('created_at', Carbon::now()->month)
                ->whereYear('created_at', Carbon::now()->year)
                ->count();

            $lastMonth = Carbon::now()->subMonth(); // Get last month date


            $subcription_last_month = UserSubscription::whereMonth('created_at', $lastMonth->month)
                ->whereYear('created_at', $lastMonth->year)
                ->count();

            $sum_of_subscription = $subcription_permonth - $subcription_last_month;

            // $subscription_persontage = round($sum_of_subscription * 100 / $totalSubscriptions);

            if ($totalSubscriptions > 0) {
                $subscription_persontage = round($sum_of_subscription * 100 / $totalSubscriptions);
            } else {
                $subscription_persontage = 0; // Or handle as needed
            }


            $subscription_vendor_permonth = UserSubscriptionHistory::whereYear('created_at', Carbon::now()->year)->sum('price');

            $lastMonth = Carbon::now()->subMonth(); // Get last month date

            $subscription_vendor_last_month = UserSubscriptionHistory::whereYear('created_at', $lastMonth->year)
                ->sum('price');

            $sum_of_subscriber = $subscription_vendor_permonth - $subscription_vendor_last_month;

            // $subscription_vendor_persontage = round($sum_of_subscriber * 100 / $totalPrice);

            if ($totalPrice > 0) {
                $subscription_vendor_persontage = round($sum_of_subscriber * 100 / $totalPrice);
            } else {
                $subscription_vendor_persontage = 0; // Or handle as needed
            }

            $sponser_permonth = GoalPaymentHistory::whereYear('created_at', Carbon::now()->year)->sum('price');

            $lastMonth = Carbon::now()->subMonth(); // Get last month date

            $sponser_last_month = GoalPaymentHistory::whereYear('created_at', $lastMonth->year)
                ->sum('price');

            $sum_of_sponser = $sponser_permonth - $sponser_last_month;

            // $sponser_persontage = round($sum_of_sponser * 100 / $totalGoal);

            if ($totalGoal > 0) {
                $sponser_persontage = round($sum_of_sponser * 100 / $totalGoal);
            } else {
                $sponser_persontage = 0; // Or handle as needed
            }


            // Year & Month
            $year = $request->get('year', date('Y'));
            $month = sprintf('%02d', $request->get('month', date('m')));
            $monthName = date('F', mktime(0, 0, 0, $month, 1));

            // Active users by day
            $activeUsers = DB::table('users')
                ->select(
                    DB::raw('DAY(created_at) as day'),
                    DB::raw("SUM(CASE WHEN role = 'user' THEN 1 ELSE 0 END) as total_users"),
                    DB::raw("SUM(CASE WHEN role = 'vendor' THEN 1 ELSE 0 END) as total_vendors")
                )
                ->where('status', 1)
                ->whereYear('created_at', $year)
                ->whereMonth('created_at', $month)
                ->groupBy('day')
                ->get()
                ->keyBy('day');

            // Prepare data per day
            $daysInMonth = cal_days_in_month(CAL_GREGORIAN, $month, $year);
            $days = [];
            $totalUsersData = [];
            $totalVendorsData = [];

            for ($day = 1; $day <= $daysInMonth; $day++) {
                $days[] = "$monthName $day";
                $totalUsersData[] = $activeUsers[$day]->total_users ?? "0";
                $totalVendorsData[] = $activeUsers[$day]->total_vendors ?? "0";
            }

            $goalPayments = DB::table('goal_payment_history')
                ->select(
                    DB::raw('MONTH(created_at) as month'),
                    DB::raw('MONTHNAME(created_at) as month_name'),
                    DB::raw('SUM(price) as total_sponsored')
                )
                ->whereYear('created_at', $year)
                ->groupBy('month', 'month_name')
                ->orderBy('month')
                ->get()
                ->keyBy('month');

            // Subscription Payment Monthly Totals
            $subscriptionPayments = DB::table('user_subscription_plan_history')
                ->select(
                    DB::raw('MONTH(created_at) as month'),
                    DB::raw('MONTHNAME(created_at) as month_name'),
                    DB::raw('SUM(price) as total_subscription')
                )
                ->whereYear('created_at', $year)
                ->groupBy('month', 'month_name')
                ->orderBy('month')
                ->get()
                ->keyBy('month');

            // Prepare 12 months data
            $monthlySponsored = [];
            $monthlySubscription = [];
            $months = [];

            for ($m = 1; $m <= 12; $m++) {
                $monthName = date('F', mktime(0, 0, 0, $m, 1));
                $months[] = $monthName;
                $monthlySponsored[] = $goalPayments[$m]->total_sponsored ?? 0;
                $monthlySubscription[] = $subscriptionPayments[$m]->total_subscription ?? 0;
            }


            // Default Currency
            $defaultCurrency = SiteSetup::first()->default_currency ?? '₹';

            // Recent 5 Services
            $service_search = $request->input('search');
            $service_data = Service::with([
                'category',
                'serviceImages',
                'vendor' => function ($query) {
                    $query->select('id', 'first_name', 'last_name', 'email', 'image');
                }
            ])
                ->when($service_search, function ($query, $search) {
                    $query->where('service_name', 'like', "%{$search}%");
                })
                ->orderBy('created_at', 'desc')
                ->limit(5)
                ->get();

            $service_data->each(function ($service) {
                $service->serviceImages->transform(function ($image) {
                    $image->service_images = asset('assets/images/service_images/' . $image->service_images);
                    return $image;
                });
            });

            $service_data->each(function ($service) {
                $avgReview = ServiceReview::where('service_id', $service->id)->avg('review_star');
                $service->avg_review = number_format($avgReview ?? 0, 1);
            });


            // Add price range
            $service_data->each(function ($record) use ($defaultCurrency) {
                $storePrices = Stores::where('service_id', $record->id)->pluck('price');
                $min = $storePrices->min() ?? 0;
                $max = $storePrices->max() ?? 0;
                $record->price_range = ($min === $max) ? "$defaultCurrency$min" : "$defaultCurrency$min - $defaultCurrency$max";
            });

            // Recent 5 Stores
            $search = $request->input('search');
            $store_data = Stores::with([
                'storeImages',
                'service' => function ($query) {
                    $query->select('id', 'service_name', 'vendor_id')
                        ->with([
                            'vendor' => function ($q) {
                                $q->select('id', 'first_name', 'last_name', 'email', 'image');
                            }
                        ]);
                }
            ])
                ->when($search, function ($query, $search) {
                    $query->where('store_name', 'like', "%{$search}%");
                })
                ->orderBy('created_at', 'desc')
                ->limit(5)
                ->get();


            $store_data->each(function ($service) {
                $service->storeImages->transform(function ($image) {
                    $image->store_images = asset('assets/images/store_images/' . $image->store_images);
                    return $image;
                });
            });


            // Recent 5 Subscribed Vendors
            $searchvendor = $request->input('search');
            $search_vendor = UserSubscription::with(['user:id,first_name,last_name,email,image'])
                ->when($searchvendor, function ($query, $searchvendor) {
                    return $query->whereHas('user', function ($query) use ($searchvendor) {
                        $query->whereRaw("CONCAT(first_name, ' ', last_name) LIKE ?", ["%{$searchvendor}%"])
                            ->orWhere('email', 'like', "%{$searchvendor}%");
                    });
                })
                ->orderBy('created_at', 'desc')
                ->limit(5)
                ->get();


            $categories = Category::select('id', 'category_name')->get();

            // Step 2: Get total number of services across all categories
            $totalServices = Service::count();

            // Step 3: Loop through each category and calculate count and percentage
            foreach ($categories as $category) {
                $categoryServiceCount = Service::where('category_id', $category->id)->count();
                $category->total_services = $categoryServiceCount;

                // Avoid division by zero
                $category->percentage_tag = $totalServices > 0
                    ? number_format(($categoryServiceCount * 100) / $totalServices, 1) . '%'
                    : '0.0%';
            }

            return response()->json([
                'status' => true,
                'message' => 'Dashboard data fetched successfully.',
                'data' => [
                    'totals' => [
                        'totalUsers' => $totalUsers,
                        'totalVendors' => $totalVendors,
                        'totalServices' => $totalServices,
                        'totalSubscriberVendors' => $totalSubscriptions,
                        'totalStores' => $totalstores,
                        'totalSubscriptionEarning' => $totalPrice,
                        'totalSponsers' => $totalGoal,
                        'user_last_month' => $user_last_month,
                        'user_persontage' => $user_persontage,
                        'store_last_month' => $store_last_month,
                        'store_persontage' => $store_persontage,
                        'service_last_month' => $service_last_month,
                        'service_persontage' => $service_persontage,
                        'subscriptionvendor_last_month' => $subcription_last_month,
                        'subsciptionvendor_persontage' => $subscription_persontage,
                        'subscription_last_month' => $subscription_vendor_last_month,
                        'subscription_persontage' => $subscription_vendor_persontage,
                        'sponser_last_month' => $sponser_last_month,
                        'sponser_persontage' => $sponser_persontage,
                    ],
                    'chartData' => [
                        'year' => $year,
                        'month' => $month,
                        'monthName' => $monthName,
                        'days' => $days,
                        'userCounts' => $totalUsersData,
                        'vendorCounts' => $totalVendorsData,
                    ],

                    'earningData' => [
                        'year' => $year,
                        'month' => $months,
                        // 'monthName' => $monthName,
                        // 'days' => $days,
                        'SponseredCounts' => $monthlySponsored,
                        'SubscriptionCounts' => $monthlySubscription,
                    ],
                    'recentServices' => $service_data,
                    'recentStores' => $store_data,
                    'recentVendors' => $search_vendor,
                    'defaultCurrency' => $defaultCurrency,
                    'categories' => $categories,
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Failed to fetch dashboard data: ' . $e->getMessage(),
            ], 500);
        }
    }



    public function loginApi(Request $request)
    {
        // Validate input
        $request->validate([
            'email' => 'required|email',
            'password' => 'required'
        ]);

        // Fetch the admin with ID = 1
        // $admin = Admin::where('id', 1)
        //     ->where('email', $request->email)
        //     ->first();

        $user = User::where('email', $request->email)
            ->first();

        $token = JWTAuth::fromUser($user);

        // Check if admin exists and password matches
        if ($user && Hash::check($request->password, $user->password)) {
            return response()->json([
                'status' => true,
                'message' => 'Login successfully',
                'admindetails' => [
                    'id' => $user->id,
                    'email' => $user->email ?? '',
                    'token' => $token,
                ]
            ]);
        }

        return response()->json([
            'status' => false,
            'message' => 'Invalid credentials or not authorized.'
        ], 401);
    }




    public function getLoginData()
    {
        // Fetch admin with ID = 1
        $admin = Admin::find(1);

        if ($admin) {
            return response()->json([
                'status' => true,
                'message' => 'Admin data fetched successfully',
                'admindetails' => [
                    'id' => $admin->id,
                    'email' => $admin->email ?? '',
                    'password' => $admin->plain_password ?? ''
                ]
            ]);
        }

        return response()->json([
            'status' => false,
            'message' => 'Admin not found'
        ], 404);
    }


    // addVendor
    public function addVendor(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $rules = [
            'first_name' => 'required',
            'last_name' => 'required',
            'email' => 'required|unique:users,email',

            'username' => 'required',
            'mobile' => 'required',
            'country_code' => 'required',
            'password' => 'required',
            'conform_password' => 'required',
            'image' => 'required',
            'status' => 'required',


        ];

        $customMessages = [
            'first_name.required' => 'Please enter the first_name.',
            'last_name.required' => 'Please enter the last_name.',

            'email.required' => 'Please enter the email.',
            'email.unique' => 'This email is already in use.',

            'username.required' => 'Please enter the username.',

            'mobile.required' => 'Please enter the mobile.',

            'country_code.required' => 'Please enter the country_code.',

            'password.required' => 'Please enter the password.',
            'password.required' => 'Please enter the password.',


            'image.required' => 'Please select the image.',
            'status.required' => 'Please enter the status.',
        ];

        $validator = Validator::make($request->all(), $rules, $customMessages);

        if ($validator->fails()) {
            return response([
                'status' => false,
                'message' => $validator->errors()->first(),
            ], 422);
        }

        $image = null;

        if ($request->hasFile('image')) {
            $originalName = pathinfo($request->image->getClientOriginalName(), PATHINFO_FILENAME);
            $sanitizedFileName = Str::slug($originalName, '_') . '.' . $request->image->getClientOriginalExtension();
            $image = time() . '_' . $sanitizedFileName;
            $request->image->move(public_path('assets/images/users_images/'), $image);
        }

        $category = new User();
        $category->first_name = $request->first_name;
        $category->last_name = $request->last_name;
        $category->email = $request->email;
        $category->username = $request->username;
        $category->mobile = $request->mobile;
        $category->country_code = $request->country_code;
        $category->status = $request->status;
        $category->password = Hash::make($request->password);

        $category->image = $image;
        $category->role = "vendor";

        $category->save();

        return response([
            'status' => true,
            'message' => 'Vedor added successfully!',
        ], 200);
    }

    public function viewCategoryold(Request $request)
    {
        $cat_id = $request->input('cat_id');

        if (!$cat_id) {
            return response([
                'status' => false,
                'message' => 'Category ID is required.',
            ], 400);
        }
        $category = Category::findOrFail($cat_id);

        // Fetch subcategories related to this category
        $subcategories = SubCategory::where('category_id', $cat_id)
            ->withCount('services') // Assuming there's a services relationship in SubCategory
            ->get();

        // Attach full image URL to each subcategory
        foreach ($subcategories as $subcategory) {
            $subcategory->subcategory_image = $subcategory->subcategory_image
                ? url('assets/images/subcategory_images/' . $subcategory->subcategory_image)
                : url('assets/images/subcategory_images/default_subcategory.png');
        }

        return response([
            'status' => true,
            'message' => 'Category View successfully!',
            'category' => $category,
            'subcategories' => $subcategories,
        ], 200);
    }

    public function viewCategory(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $cat_id = $request->input('cat_id');

        if (!$cat_id) {
            return response([
                'status' => false,
                'message' => 'Category ID is required.',
            ], 400);
        }

        // $categories = Category::findOrFail($cat_id);

        $categories = Category::where('id', $cat_id)->get();


        foreach ($categories as $category) {
            $category->category_icon = $category->category_icon
                ? url('assets/images/category_icons/' . $category->category_icon)
                : url('assets/images/category_icons/default_subcategory.png');
            $category->category_image = $category->category_image
                ? url('assets/images/category_images/' . $category->category_image)
                : url('assets/images/category_images/default_subcategory.png');
        }


        // Fetch subcategories under the category
        $subcategories = SubCategory::where('category_id', $cat_id)->get();

        // Process each subcategory
        foreach ($subcategories as $subcategory) {
            // Image formatting
            $subcategory->subcategory_image = $subcategory->subcategory_image
                ? url('assets/images/subcategory_images/' . $subcategory->subcategory_image)
                : url('assets/images/subcategory_images/default_subcategory.png');

            // Service count using FIND_IN_SET
            $subcategory->services_count = DB::table('services')
                ->whereRaw("FIND_IN_SET(?, subcategory_id)", [$subcategory->id])
                ->count();
        }

        return response([
            'status' => true,
            'message' => 'Category View successfully!',
            'category' => $category,
            'subcategories' => $subcategories,
        ], 200);
    }

    public function reportTextList(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $search = $request->input('search');
        $page = (int) $request->input('page', 1);
        $pageSize = (int) $request->input('pageSize', 10);
        $orderColumn = $request->input('orderColumn', 'created_at');
        $orderDir = strtoupper($request->input('orderDir', 'DESC'));

        $allowedOrderColumns = ['created_at', 'text']; // define allowed columns

        // Validate orderColumn and orderDir
        if (!in_array($orderColumn, $allowedOrderColumns)) {
            $orderColumn = 'created_at';
        }

        if (!in_array($orderDir, ['ASC', 'DESC'])) {
            $orderDir = 'DESC';
        }

        $query = ReportText::select('id', 'text', 'created_at');

        if (!empty($search)) {
            $query->where('text', 'like', '%' . $search . '%');
        }

        $total = $query->count();

        $data = $query->orderBy($orderColumn, $orderDir)
            ->skip(($page - 1) * $pageSize)
            ->take($pageSize)
            ->get();

        if ($data->isNotEmpty()) {
            return response()->json([
                'status' => true,
                'message' => 'Report Text found',
                'data' => $data,
                'Pagination' => [
                    'total_pages' => ceil($total / $pageSize),
                    'total_records' => $total,
                    'current_page' => $page,
                    'records_per_page' => $pageSize,
                ]
            ]);
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Report Text not found',
            ]);
        }
    }

    public function getUserview(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        // Fetch the user by ID
        $user_id = $request->input('user_id');
        // $user = User::find($user_id);

        $user = User::select('id', 'first_name', 'last_name', 'email', 'image', 'mobile') // choose only needed fields
            ->find($user_id);

        $user->first_name = $user->first_name ?? "";
        $user->last_name = $user->last_name ?? "";
        $user->email = $user->email ?? "";
        $user->mobile = $user->mobile ?? "";

        // $user->first_name = $user->first_name ?? "";


        if (!$user) {
            return response()->json([
                'status' => false,
                'message' => 'User not found',
            ], 404);
        }

        // Count total reviews and likes
        $totalReviews = ServiceReview::where('user_id', $user_id)->count();
        $totalLikes = ServiceLike::where('user_id', $user_id)->count();

        // Fetch review data
        $reviews = ServiceReview::with(['service', 'service.serviceImages'])
            ->where('user_id', $user_id)
            ->get()
            ->map(function ($review) {
                $serviceImages = [];

                if ($review->service && $review->service->serviceImages) {
                    $serviceImages = $review->service->serviceImages->pluck('service_images')->map(function ($img) {
                        return url('assets/images/service_images/' . $img);
                    })->toArray();

                    if ($review->service) {
                        $prices = Stores::where('service_id', $review->service->id)->pluck('price');
                        $min = $prices->min() ?? 0;
                        $max = $prices->max() ?? 0;

                        $defaultCurrency = SiteSetup::first()->default_currency ?? '$';

                        $priceRange = ($min === $max)
                            ? "$defaultCurrency$min"
                            : "$defaultCurrency$min - $defaultCurrency$max";
                    }
                }

                return [
                    'created_at' => $review->created_at,
                    'service_id' => $review->service_id,
                    'service_name' => $review->service->service_name ?? '',
                    'service_images' => $serviceImages,
                    'price_range' => $priceRange,
                    'review_star' => $review->review_star ?? 0,
                    'review_message' => $review->review_message ?? '',
                ];
            });

        // Fetch like data
        $likes = ServiceLike::with(['service', 'service.serviceImages'])
            ->where('user_id', $user_id)
            ->get()
            ->map(function ($like) {
                $serviceImages = [];

                if ($like->service && $like->service->serviceImages) {
                    $serviceImages = $like->service->serviceImages->pluck('service_images')->map(function ($img) {
                        return url('assets/images/service_images/' . $img);
                    })->toArray();
                }

                if ($like->service) {
                    $prices = Stores::where('service_id', $like->service->id)->pluck('price');
                    $min = $prices->min() ?? 0;
                    $max = $prices->max() ?? 0;

                    $defaultCurrency = SiteSetup::first()->default_currency ?? '$';

                    $priceRange = ($min === $max)
                        ? "$defaultCurrency$min"
                        : "$defaultCurrency$min - $defaultCurrency$max";
                }

                return [
                    'created_at' => $like->created_at,
                    'service_id' => $like->service_id,
                    'service_name' => $like->service->service_name ?? '',
                    'service_images' => $serviceImages,
                    'price_range' => $priceRange,
                ];
            });

        // Return the API response
        return response()->json([
            'status' => true,
            'message' => 'User data fetched successfully',
            'user' => $user,
            'total_reviews' => $totalReviews,
            'total_likes' => $totalLikes,
            'reviews' => $reviews,
            'likes' => $likes,
        ]);
    }

    public function editService(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            // $vendor_id = $request->vendor_id;
            $service_id = $request->service_id;

            // Check if vendor_id is provided
            // if (!$vendor_id) {
            //     return response([
            //         'status' => false,
            //         'message' => 'Vendor ID is required.',
            //     ], 404);
            // }

            // Find the service
            $service = Service::find($service_id);

            // Check for existence and authorization
            if (!$service) {
                return response([
                    'status' => false,
                    'message' => 'Service is not available.',
                ], 404);
            }


            $updateData = $request->only([
                'service_name',
                'service_description',
                'category_id',
                'subcategory_id',
                'service_website',
                'service_country_code',
                'service_phone',
                'address',
                'lat',
                'lon',
                'open_days',
                'service_address',
                'closed_days',
                'open_time',
                'close_time',
                'instagram_link',
                'facebook_link',
                'twitter_link',
                'whatsapp_link',
                'aspect_ratio',
                'employee_strength',
                'published_month',
                'published_year',
                'service_email',
                'area',
                'city',
                'state',
                'country',
                'cover_image',
                'meta_title',
                'meta_description',
                'video_url',
                'vendor_id',
            ]);

            // If closed_days is an empty string or not provided, set it to null
            if (empty($request->closed_days)) {
                $updateData['closed_days'] = null;
            }

            // Update the service
            $service->update($updateData);

            // Handle service images
            if ($request->hasFile('service_images')) {
                foreach ($request->file('service_images') as $file) {
                    $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                    $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                    $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                    $file->move(public_path('assets/images/service_images'), $fileName);

                    $service->serviceImages()->create([
                        'service_images' => $fileName,
                    ]);
                }
            }

            // Handle video thumbnail upload and update column
            if ($request->hasFile('video_thumbnail')) {
                $file = $request->file('video_thumbnail');
                $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                $file->move(public_path('assets/images/video_thumbnail'), $fileName);

                $service->update([
                    'video_thumbnail' => $fileName,
                ]);
            }

            // Handle video upload and update column
            if ($request->hasFile('video')) {
                $file = $request->file('video');
                $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                $file->move(public_path('assets/images/service_video'), $fileName);

                $service->update([
                    'video' => $fileName,
                ]);
            }

            // Handle cover image upload and update column
            if ($request->hasFile('cover_image')) {
                $file = $request->file('cover_image');
                $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                $file->move(public_path('assets/images/cover_images'), $fileName);

                $service->update([
                    'cover_image' => $fileName,
                ]);
            }

            // Get service images and video (with full paths)
            // Get service images with ID and full paths
            $serviceImages = $service->serviceImages()->get(['id', 'service_images'])->map(function ($image) {
                return [
                    'id' => $image->id,
                    'service_images' => asset('assets/images/service_images/' . $image->service_images), // Full path
                ];
            });


            $serviceVideo = $service->video ? asset('assets/images/service_video/' . $service->video) : '';

            $serviceCoverimage = $service->cover_image ? asset('assets/images/cover_images/' . $service->cover_image) : '';

            $serviceVideoThumb = $service->video_thumbnail ? asset('assets/images/video_thumbnail/' . $service->video_thumbnail) : '';


            // Replace null values with empty strings
            $service = $service->toArray();
            array_walk_recursive($service, function (&$value) {
                if (is_null($value)) {
                    $value = '';
                }
            });
            $vendor = User::select('first_name', 'last_name')
                ->where('id', $service['vendor_id'] ?? 0)
                ->first();

            $vendorName = $vendor ? trim($vendor->first_name . ' ' . $vendor->last_name) : '';

            $service['vendor_firstname'] = $vendor ? $vendor->first_name : '';
            $service['vendor_lastname'] = $vendor ? $vendor->last_name : '';

            // Return response with updated data, including service_images and service_video
            return response([
                'status' => true,
                'message' => 'Service updated successfully!',
                'service' => $service,  // Include the updated service data with empty strings instead of null
                'service_images' => $serviceImages,  // Include service images with full paths
                'video' => $serviceVideo,  // Include service video with full path
                'cover_image' => $serviceCoverimage,  // Include service video with full path
                'video_thumbnail' => $serviceVideoThumb,  // Include service video with full path
                // 'vendor_firstname' => $vendor->first_name,
                // 'vendor_lastname' => $vendor->last_name,


            ], 200);
        } catch (\Exception $e) {
            return response([
                'status' => false,
                'message' => 'An error occurred while updating the service: ' . $e->getMessage(),
            ], 500);
        }
    }
    public function getVendorView(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        // Fetch the vendor user by ID
        $user_id = $request->input('user_id');
        // $user = User::find($user_id);


        $user = User::select('id', 'first_name', 'last_name', 'email', 'image', 'country_code', 'mobile') // include both fields
            ->find($user_id);

        if (!$user) {
            return response()->json([
                'status' => false,
                'message' => 'Vendor not found',
            ], 404);
        }

        $user->full_mobile = trim($user->country_code . ' ' . $user->mobile);

        // Default currency
        $defaultCurrency = SiteSetup::first()->default_currency ?? '$';

        // Stores associated with vendor's services
        // $stores = Stores::whereHas('service', function ($query) use ($user_id) {
        //     $query->where('vendor_id', $user_id);
        // })->with('service', 'subcategory')->get();

        // $stores = Stores::whereHas('service', function ($query) use ($user_id) {
        //     $query->where('vendor_id', $user_id);
        // })
        //     ->with([
        //         'service:id,id,service_name', // Only load needed fields from Service
        //         'subcategory:id,id,subcategory_name' // Optional: limit subcategory fields too
        //     ])
        //     ->get();

        $stores = Stores::whereHas('service', function ($query) use ($user_id) {
            $query->where('vendor_id', $user_id);
        })
            ->with([
                'service:id,id,service_name', // Only fetch minimal service data (if needed temporarily)
                'subcategory:id,id,subcategory_name'
            ])
            ->select('id', 'store_name', 'price', 'service_id', 'subcategory_id', 'created_at') // Only fetch needed store fields
            ->get();

        $stores->transform(function ($store) {
            $store->store_images = $store->storeImages->map(function ($img) {
                return [
                    'id' => $img->id,
                    'url' => url('assets/images/store_images/' . $img->store_images),
                ];
            })->toArray(); // Ensure it's an array

            $store->subcategory_name = $store->subcategory->subcategory_name ?? '';

            unset($store->storeImages); // Remove the original relationship data

            return $store;
        });

        // $subcategory = Subcategory::select('subcategory_name')
        //     ->where('id', $stores['subcategory_id'])
        //     ->first();
        // $store->subcategory_name = $subcategory ? $subcategory->subcategory_name : '';
        $totalStoreCount = $stores->count();

        // Services by vendor
        $services = Service::select('id', 'service_name', 'vendor_id', 'category_id')
            ->with([
                'category:id,id,category_name', // or select the required fields from category
                'serviceImages',
                'vendor:id,first_name,last_name,email,image',
            ])
            ->where('vendor_id', $user_id)
            ->orderBy('created_at', 'desc')
            ->get();

        // $serviceImages = ServiceImages::where('service_id', $service->id)->get();
        // $serviceDetail['service_images'] = $serviceImages->isNotEmpty()
        //     ? $serviceImages->map(function ($image) {
        //         return url('assets/images/service_images/' . $image->service_images);
        //     })
        //     : [url('assets/images/service_images/default_service.png')];

        $services->transform(function ($service) {
            $service->service_images = $service->serviceImages->map(function ($img) {
                return [
                    'id' => $img->id,
                    'url' => url('assets/images/service_images/' . $img->service_images),
                ];
            })->toArray(); // Ensure it's an array

            unset($service->serviceImages); // Remove the original relationship data

            return $service;
        });

        // Add price range
        $services->transform(function ($service) use ($defaultCurrency) {
            $prices = Stores::where('service_id', $service->id)->pluck('price');
            $min = $prices->min() ?? 0;
            $max = $prices->max() ?? 0;
            $service->price_range = $min === $max
                ? "$defaultCurrency$min"
                : "$defaultCurrency$min - $defaultCurrency$max";
            return $service;
        });

        // Reviews of vendor's services
        $reviews = ServiceReview::with([
            'user:id,first_name,last_name,email,image',
            'service:id,service_name,service_description',
        ])
            ->whereHas('service', function ($query) use ($user_id) {
                $query->where('vendor_id', $user_id);
            })
            ->orderBy('created_at', 'desc')
            ->get(); // Removed ->paginate(10)

        // Format created_at
        $reviews->transform(function ($review) {
            $review->formatted_created_at = $review->created_at
                ? $review->created_at->format('d M, Y / g:i A')
                : '';
            return $review;
        });

        // Totals
        $totalReviewCount = ServiceReview::whereHas('service', function ($q) use ($user_id) {
            $q->where('vendor_id', $user_id);
        })->count();

        $totalLikesCount = ServiceLike::whereHas('service', function ($q) use ($user_id) {
            $q->where('vendor_id', $user_id);
        })->count();

        $totalLeadCount = ServiceLead::whereHas('service', function ($q) use ($user_id) {
            $q->where('vendor_id', $user_id);
        })->count();

        // Subscriptions
        // $subscriptions = UserSubscription::where('user_id', $user_id)
        //     ->with('user:id,first_name,last_name,email,image')
        //     ->orderBy('created_at', 'desc')
        //     ->get();

        // Subscriptions
        $subscriptions = UserSubscription::select('id', 'user_id', 'plan_name', 'price', 'start_date', 'expire_date', 'payment_mode', 'status', 'created_at')->where('user_id', $user_id)
            ->with('user:id,first_name,last_name,email,image')
            ->orderBy('created_at', 'desc')
            ->get();

        // Sponsorships
        // $sponsors = GoalPayment::whereHas('goal.campaign.vendor', function ($q) use ($user_id) {
        //     $q->where('id', $user_id);
        // })
        //     ->with(['goal.campaign.vendor', 'goal', 'goal.campaign', 'service', 'vendor'])
        //     ->orderBy('created_at', 'desc')
        //     ->get();

        $sponsors = GoalPayment::whereHas('goal.campaign.vendor', function ($q) use ($user_id) {
            $q->where('id', $user_id);
        })
            ->with([
                // 'goal.campaign.vendor',
                'goal',
                'goal.campaign',
                'service' => function ($q) {
                    $q->select('id', 'service_name'); // Only required fields
                },
                'vendor' => function ($q) {
                    $q->select('id', 'username'); // Only required fields
                }
            ])
            ->orderBy('created_at', 'desc')
            ->get();

        // Build response
        return response()->json([
            'status' => true,
            'message' => 'Vendor data fetched successfully',
            'vendor' => $user,
            'default_currency' => $defaultCurrency,
            'total_store_count' => $totalStoreCount,
            'total_review_count' => $totalReviewCount,
            'total_like_count' => $totalLikesCount,
            'total_lead_count' => $totalLeadCount,
            'stores' => $stores,
            'services' => $services,
            'reviews' => $reviews,
            'subscriptions' => $subscriptions,
            'sponsors' => $sponsors,
        ]);
    }

    // updateStore
    public function editStore(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        try {
            $store_id = $request->store_id;

            // Validate store_id parameter
            if (!$store_id) {
                return response([
                    'status' => false,
                    'message' => 'Store ID is required',
                ], 400);
            }

            // Find the store by store_id
            $store = Stores::find($store_id);

            if (!$store) {
                return response([
                    'status' => false,
                    'message' => 'Store not found',
                ], 404);
            }

            // Update the store details
            $store->update($request->only([
                'store_name',
                'store_description',
                'price',
                'subcategory_id',
            ]));

            // Handle store images
            if ($request->hasFile('store_images')) {
                foreach ($request->file('store_images') as $file) {
                    $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                    $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                    $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                    $file->move(public_path('assets/images/store_images'), $fileName);

                    // Save image record in the store_images table
                    $store->storeImages()->create([
                        'store_images' => $fileName,
                    ]);
                }
            }

            // Handle store attachments
            if ($request->hasFile('store_attachments')) {
                foreach ($request->file('store_attachments') as $file) {
                    $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                    $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                    $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                    $file->move(public_path('assets/images/store_attachments'), $fileName);

                    // Save attachment record in the store_attachments table
                    $store->storeAttachments()->create([
                        'store_attachments' => $fileName,
                    ]);
                }
            }

            // Fetch subcategory name
            $subcategory = SubCategory::find($store->subcategory_id);
            $subcategoryName = $subcategory ? $subcategory->subcategory_name : null;

            // Format store data with full image and attachment paths
            $storeData = $store->only(['id', 'service_id', 'store_name', 'store_description', 'subcategory_id', 'price', 'created_at', 'updated_at']);
            $storeData['subcategory_name'] = $subcategoryName;


            // Add store images
            $storeData['storeImages'] = $store->storeImages->map(function ($image) {
                return [
                    'id' => $image->id,
                    'store_images' => asset('assets/images/store_images/' . $image->store_images),
                ];
            });

            // Add store attachments
            $storeData['storeAttachments'] = $store->storeAttachments->map(function ($attachment) {
                return [
                    'id' => $attachment->id,
                    'store_attachments' => asset('assets/images/store_attachments/' . $attachment->store_attachments),
                ];
            });

            return response([
                'status' => true,
                'message' => 'Store updated successfully!',
                'store' => $storeData,
            ], 200);
        } catch (\Exception $e) {
            return response([
                'status' => false,
                'message' => 'An error occurred while updating the store: ' . $e->getMessage(),
            ], 500);
        }
    }
    public function deleteReview(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $review_id = $request->id;

        // Find the service review by review_id and ensure it belongs to the user
        $serviceReview = ServiceReview::where('id', $review_id)
            ->first();

        // Check if the review exists and belongs to the user
        if (!$serviceReview) {
            return response()->json(
                [
                    'status' => false,
                    'message' => 'Review not found or not authorized',
                ],
                404
            );
        }
        // Delete the review
        $serviceReview->delete();

        return response()->json(
            [
                'status' => true,
                'message' => 'User review deleted successfully',
            ],
            200
        );
    }

    public function viewService_old(Request $request)
    {
        $service_id = $request->service_id;
        $service = Service::with('vendor')->findOrFail($service_id); // Fetch service along with vendor details

        $defaultCurrency = SiteSetup::first()->default_currency;


        $stores = Stores::whereHas('service', function ($query) use ($service_id) {
            $query->where('id', $service_id);
        })
            // ->with([
            //     'service:id,id,service_name', // Only fetch minimal service data (if needed temporarily)

            // ])
            ->select('id', 'store_name', 'price', 'service_id', 'subcategory_id', 'created_at') // Only fetch needed store fields
            ->get();

        $stores->transform(function ($store) {
            $store->store_images = $store->storeImages->map(function ($img) {
                return [
                    'id' => $img->id,
                    'url' => url('assets/images/store_images/' . $img->store_images),
                ];
            })->toArray(); // Ensure it's an array

            // $store->subcategory_name = $store->subcategory->subcategory_name ?? '';

            unset($store->storeImages); // Remove the original relationship data

            return $store;
        });



        // Build response
        return response()->json([
            'status' => true,
            'message' => 'Vendor data fetched successfully',
            'service' => [
                'id' => $service->id,
                'service_name' => $service->service_name ?? '',
                'vendor' => [
                    'id' => $service->vendor->id ?? '',
                    'first_name' => $service->vendor->first_name ?? '',
                    'last_name' => $service->vendor->last_name ?? '',
                    'email' => $service->vendor->email ?? '',
                    'image' => $service->vendor->image
                        ? url('assets/images/user/' . $service->vendor->image)
                        : '',
                ],
            ],
            'stores' => $stores,
        ]);
    }

    public function viewServiceold(Request $request)
    {
        $service_id = $request->service_id;
        $search = $request->input('search'); // optional search input

        $service = Service::with('vendor')->findOrFail($service_id); // Fetch service with vendor

        $defaultCurrency = SiteSetup::first()->default_currency;

        $storesQuery = Stores::whereHas('service', function ($query) use ($service_id) {
            $query->where('id', $service_id);
        });

        if (!empty($search)) {
            $storesQuery->where('store_name', 'like', '%' . $search . '%');
        }

        $stores = $storesQuery
            ->select('id', 'store_name', 'price', 'service_id', 'subcategory_id', 'created_at')
            ->get();

        $stores->transform(function ($store) {
            $store->store_images = $store->storeImages->map(function ($img) {
                return [
                    'id' => $img->id,
                    'url' => url('assets/images/store_images/' . $img->store_images),
                ];
            })->toArray();

            unset($store->storeImages); // Clean relationship

            return $store;
        });

        return response()->json([
            'status' => true,
            'message' => 'Vendor data fetched successfully',
            'service' => [
                'id' => $service->id,
                'service_name' => $service->service_name ?? '',
                'vendor' => [
                    'id' => $service->vendor->id ?? '',
                    'first_name' => $service->vendor->first_name ?? '',
                    'last_name' => $service->vendor->last_name ?? '',
                    'email' => $service->vendor->email ?? '',
                    'image' => $service->vendor->image
                        ? url('assets/images/user/' . $service->vendor->image)
                        : '',
                ],
                'stores' => $stores,
            ],

        ]);
    }

    public function viewService(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        $service_id = $request->service_id;
        $search = $request->input('search'); // optional search input
        $perPage = $request->input('per_page', 10); // default to 10 items per page

        $service = Service::with('vendor')->findOrFail($service_id); // Fetch service with vendor

        $defaultCurrency = SiteSetup::first()->default_currency;

        $storesQuery = Stores::whereHas('service', function ($query) use ($service_id) {
            $query->where('id', $service_id);
        });

        if (!empty($search)) {
            $storesQuery->where('store_name', 'like', '%' . $search . '%');
        }

        $storesPaginated = $storesQuery
            ->select('id', 'store_name', 'price', 'service_id', 'subcategory_id', 'created_at')
            ->with('storeImages')
            ->orderByDesc('created_at')
            ->paginate($perPage);

        $stores = $storesPaginated->getCollection()->transform(function ($store) {
            $store->store_images = $store->storeImages->map(function ($img) {
                return [
                    'id' => $img->id,
                    'url' => url('assets/images/store_images/' . $img->store_images),
                ];
            })->toArray();

            unset($store->storeImages);
            return $store;
        });

        return response()->json([
            'status' => true,
            'message' => 'Vendor data fetched successfully',
            'service' => [
                'id' => $service->id,
                'service_name' => $service->service_name ?? '',
                'vendor' => [
                    'id' => $service->vendor->id ?? '',
                    'first_name' => $service->vendor->first_name ?? '',
                    'last_name' => $service->vendor->last_name ?? '',
                    'email' => $service->vendor->email ?? '',
                    'image' => $service->vendor->image
                        ? url('assets/images/user/' . $service->vendor->image)
                        : '',
                ],
                'stores' => $stores,
            ],
            'pagination' => [
                'current_page' => $storesPaginated->currentPage(),
                'last_page' => $storesPaginated->lastPage(),
                'per_page' => $storesPaginated->perPage(),
                'total' => $storesPaginated->total(),
            ],
        ]);
    }


    public function viewStore(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        $service_id = $request->service_id;
        $search = $request->input('search');

        if (!$service_id) {
            return response()->json([
                'status' => false,
                'message' => 'Service ID is required.',
            ], 400);
        }

        // Fetch the main service with vendor relationship
        $service = Service::with('vendor')->find($service_id);

        if (!$service) {
            return response()->json([
                'status' => false,
                'message' => 'Service not found.',
            ], 404);
        }

        $defaultCurrency = SiteSetup::first()->default_currency ?? 'USD';

        // Prepare store query
        $storesQuery = Service::query()
            ->where('service_name', $service->service_name);

        if (!empty($search)) {
            $storesQuery->where('service_name', 'like', '%' . $search . '%');
        }

        // Get only one store
        $store = $storesQuery
            ->with('serviceImages')
            ->select('id', 'service_name', 'category_id', 'subcategory_id', 'lat', 'lon', 'employee_strength', 'open_days', 'closed_days', 'open_time', 'close_time', 'published_month', 'published_year', 'service_address', 'created_at', 'service_description', 'video_url', 'address')
            ->first();

        $store->video_url = $store->video_url ?? "";

        $category = Category::where('id', $store->category_id)->first();
        $store->category = $category->category_name;

        $service_count = Service::where('vendor_id', $service->vendor_id)->count();
        $store->total_service_count = $service_count;


        $is_done = GoalPayment::where('service_id', $service_id)
            ->whereDate('end_date_do', '>=', now()->format('Y-m-d'))
            ->exists();

        $store->is_feature = $is_done ? "1" : "0";


        $totalReviews = ServiceReview::where('service_id', $service->id)->get();
        $totalReviewStars = $totalReviews->sum('review_star');
        $totalReviewCount = $totalReviews->count();

        $store->totalReviewCount = $totalReviewCount;
        $store->totalAvgReview =
            $totalReviewCount > 0 ? (string) round($totalReviewStars / $totalReviewCount, 1) : '';

        // Add image transformation
        if ($store) {
            $store->service_images = $store->serviceImages->map(function ($img) {
                return [
                    'id' => $img->id,
                    'url' => url('assets/images/service_images/' . $img->service_images),
                ];
            })->toArray();

            unset($store->serviceImages);
        }

        if ($service->published_month && $service->published_year) {
            try {
                // Convert month name to month number
                $monthNumber = \Carbon\Carbon::parse("01-{$service->published_month}-2000")->month;
                $publishedDate = \Carbon\Carbon::createFromDate($service->published_year, $monthNumber, 1);
                $totalYearsCount = $publishedDate->diffInYears(now());
            } catch (\Exception $e) {
                $totalYearsCount = 0; // Default to 0 if there's an error
            }
        } else {
            $totalYearsCount = 0; // Default to 0 if month or year is missing
        }
        $store->total_years_count = $totalYearsCount;

        $follow = Service::select('instagram_link', 'facebook_link', 'whatsapp_link', 'twitter_link')->where('id', $service_id)
            ->first();

        $follow->instagram_link = $follow->instagram_link ?? "";
        $follow->facebook_link = $follow->facebook_link ?? "";
        $follow->whatsapp_link = $follow->whatsapp_link ?? "";
        $follow->twitter_link = $follow->instagram_link ?? "";

        $goalPayments = GoalPayment::where('service_id', $service_id)
            ->select('goal_id', 'payment_mode')
            ->get()
            ->keyBy('goal_id'); // Index by goal_id for quick lookup

        // Fetch goal records using the goal_ids
        $goals = Goals::whereIn('id', $goalPayments->keys())->get();

        if ($goals->isNotEmpty()) {
            foreach ($goals as $goal) {

                $payment = $goalPayments[$goal->id] ?? null;

                $campaign = Campaign::where('id', $goal->campaign_id)->first();
                $goal->campaign_name = $campaign ? $campaign->campaign_name : "";
                $goal->address = $campaign ? $campaign->address : "";
                $goal->area_distance = $campaign ? $campaign->area_distance : "";
                $time_stamp = now()->format('Y-m-d');


                $done = Goals::where('end_date', '>=', $time_stamp)->first();
                $goal->status = $done ? "1" : "0";



                $goal->payment_mode = $payment ? $payment->payment_mode : "";
                // $goal->payment_mode = $goalIds->payment_mode;
                $goal->start_date = $goal->start_date ?? "";
                $goal->end_date = $goal->end_date ?? "";
                $goal->days = $goal->days ?? "";
                $goal->price = $goal->price ?? "";
            }
        }


        return response()->json([
            'status' => true,
            'message' => 'Vendor data fetched successfully',
            'service' => [
                'id' => $service->id,
                'service_name' => $service->service_name ?? '',
                'vendor' => [
                    'id' => $service->vendor->id ?? '',
                    'first_name' => $service->vendor->first_name ?? '',
                    'last_name' => $service->vendor->last_name ?? '',
                    'email' => $service->vendor->email ?? '',
                    'image' => $service->vendor->image
                        ? url('assets/images/users_images/' . $service->vendor->image)
                        : '',
                    'mobile' => $service->vendor->mobile ?? '',
                    'username' => $service->vendor->username ?? '',
                ],
                'stores' => $store,
                'follow_us' => $follow,
                'GoalData' => $goals,
            ],
        ]);
    }

    public function frontedSlider(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            // Always update ID = 1
            $siteSettings = IndexSlider::find(1);

            if (!$siteSettings) {
                return response()->json([
                    'status' => false,
                    'message' => 'Index Slider not found.',
                ], 404);
            }

            // Fields to update if provided
            $fields = [
                'title',
                'body',
                'link',
            ];

            foreach ($fields as $field) {
                if ($request->has($field)) {
                    $siteSettings->$field = $request->$field;
                }
            }

            // Image fields
            $imageFields = [
                'image' => 'image',
            ];

            foreach ($imageFields as $inputName => $columnName) {
                if ($request->hasFile($inputName)) {
                    $file = $request->file($inputName);
                    $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                    $sanitizedFileName = Str::slug($originalName, '_') . '.' . $file->getClientOriginalExtension();
                    $fileName = time() . '_' . uniqid() . '_' . $sanitizedFileName;

                    $file->move(public_path('assets/images/index'), $fileName);

                    $siteSettings->$columnName = $fileName;
                }
            }

            $siteSettings->save();

            return response()->json([
                'status' => true,
                'message' => 'Fronted Slider updated successfully!',
                'data' => [
                    'id' => $siteSettings->id,
                    'title' => $siteSettings->title ?? '',
                    'body' => $siteSettings->body ?? '',
                    'link' => $siteSettings->link ?? '',
                    'image' => $siteSettings->image
                        ? asset('assets/images/index/' . $siteSettings->image)
                        : '',
                ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating general settings: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function uniqueServiceCities()
    {
        $user = Auth::user();
        $user_id = $user->id;
        // Fetch all distinct non-null city values
        $cities = DB::table('services')
            ->whereNotNull('city')
            ->distinct()
            ->pluck('city')
            ->filter() // remove empty strings/nulls
            ->values(); // reset array keys

        return response()->json([
            'status' => true,
            'message' => 'Unique cities fetched successfully',
            'cities' => $cities,
        ]);
    }

    public function allPaymentGatewayKey(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $canceledBookings = [];
        $category = PaymentGatewayKey::get();

        foreach ($category as $category_done) {

            $result["id"] = (string) $category_done->id;
            $result["status"] = (string) $category_done->status;

            $result["text"] = (string) $category_done->text;


            $result["public_key"] = (string) $category_done->public_key;

            $result["secret_key"] = (string) $category_done->secret_key;

            $result["mode"] = (string) $category_done->mode;

            $result["country_code"] = (string) $category_done->country_code ?? "";

            $result["currency_code"] = (string) $category_done->currency_code ?? "";

            $canceledBookings[] = $result;
        }

        return response()->json([
            'status' => true,
            'message' => 'Payment Gateway Key Done',
            'all_payment' => $canceledBookings,
        ]);
    }
    // updateUserProfile
    public function updateAdminProfile(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        // $user_id = $request->user_id;
        $user = User::find($user_id);

        if (!$user) {
            return response()->json(
                [
                    'status' => false,
                    'message' => 'User not found',
                ],
                404
            );
        }

        // Handle image upload if provided
        if ($request->hasFile('image')) {
            $file = $request->file('image');
            $image_exts = ['tif', 'jpg', 'jpeg', 'gif', 'png'];
            $extension = $file->getClientOriginalExtension();

            if (!in_array($extension, $image_exts)) {
                return response()->json(
                    [
                        'status' => false,
                        'message' => 'Image Type Error',
                    ],
                    400
                );
            }

            // Delete the old image if it exists
            $oldImagePath = public_path('/assets/images/users_images/' . $user->image);
            if ($user->image && file_exists($oldImagePath)) {
                unlink($oldImagePath);
            }

            // Save the new image
            $fileName = 'image-' . uniqid() . '.' . $extension;
            $file->move(public_path('/assets/images/users_images/'), $fileName);
            $image = $fileName;
        } else {
            $image = $user->image;
        }


        // Hash the password if provided
        $password = $request->has('password') ? Hash::make($request->input('password')) : $user->password;

        // Update user profile data
        $user->update([
            'first_name' => $request->input('first_name', $user->first_name),
            'last_name' => $request->input('last_name', $user->last_name),
            'mobile' => $request->input('mobile', $user->mobile),
            'email' => $request->input('email', $user->email),
            'username' => $request->input('username', $user->username),
            'password' => $password,
            'image' => $image,
        ]);

        // Return updated user details with the full image URL or default image if none exists
        return response()->json(
            [
                'status' => true,
                'message' => 'Profile updated successfully ...!',
                'admindetails' => [
                    'id' => $user->id,
                    'first_name' => $user->first_name ?? '',
                    'last_name' => $user->last_name ?? '',
                    'mobile' => $user->mobile ?? '',
                    'email' => $user->email ?? '',
                    // 'password' => $user->password ?? '',
                    'image' => $user->image
                        ? url('assets/images/users_images/' . $user->image)
                        : "",
                ],
            ],
            200
        );
    }
    public function changePassword(Request $request)
    {
        // $user_id = $request->user_id;
        $user = Auth::user();
        $user_id = $user->id;

        if (!Admin::where('id', $user_id)->exists()) {
            // return response()->json(['error' => "Invalid User..!"]);
            return response()->json(
                [
                    'status' => false,
                    'message' => 'User not found',
                ],
                404
            );
        }

        if (!empty($request->password)) {
            $validator = Validator::make($request->all(), [
                'password' => 'required',
                'npassword' => 'required', // Adjust the minimum length as needed
                'cpassword' => 'required|same:npassword',
            ]);

            if ($validator->fails()) {
                // return $this->sendError($validator->errors());
                return response()->json(
                    [
                        'status' => false,
                        'message' => 'new password and confirm password not match.',
                    ],
                    404
                );
            }

            $user = Admin::find($user_id);

            // if ($user && $user->password == md5($request->password)) {
            if ($user && Hash::check($request->password, $user->password)) {

                $user->update(['password' => bcrypt($request->npassword), 'main_password' => $request->npassword]);

                return response()->json(
                    [
                        'status' => true,
                        'message' => 'Password updated successfully ...!',
                    ],
                    200
                );
            } else {
                // return $this->sendError('Invalid Password.');

                return response()->json([
                    'status' => false,
                    'message' => 'Invalid Password.',
                    // 'all_payment' => $canceledBookings,
                ]);
            }
        }

        // return $this->sendError('Password field is required.');
        return response()->json(
            [
                'status' => false,
                'message' => 'User not found',
            ],
            404
        );
    }

    public function updateFrontedStatus(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        try {
            $subscription_id = $request->id;

            if (!$subscription_id) {
                return response()->json([
                    'status' => false,
                    'message' => 'Home Settings ID is required.',
                ], 400);
            }

            $subscription = HomeSettings::find($subscription_id);

            if (!$subscription) {
                return response()->json([
                    'status' => false,
                    'message' => 'Home Settings not found.',
                ], 404);
            }

            // Update fields if present in the request
            if ($request->has('status')) {
                $subscription->status = $request->status;
            }

            $subscription->save();

            return response()->json([
                'status' => true,
                'message' => 'Home Settings updated successfully!',
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating the subscription: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function updateReview(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        // if (!$request->has('user_id')) {
        //     return response()->json(
        //         [
        //             'status' => false,
        //             'message' => 'User ID is required',
        //         ],
        //         400
        //     );
        // }

        // $user_id = $request->user_id;
        $id = $request->id;

        $user = ServiceReview::find($id);

        // Update user profile data
        $user->update([
            'service_id' => $request->input('service_id', $user->service_id),
            'review_star' => $request->input('review_star', $user->review_star),
            'review_message' => $request->input('review_message', $user->review_message),
        ]);

        // Return updated user details with the full image URL or default image if none exists
        return response()->json(
            [
                'status' => true,
                'message' => 'Profile updated successfully ...!',
                'admindetails' => [
                    'id' => $user->id,
                    'service_id' => $user->service_id ?? '',
                    'review_star' => $user->review_star ?? '',
                    'review_message' => $user->review_message ?? '',
                ],
            ],
            200
        );
    }
    public function updateSlider(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        $slider_id = $request->id;
        $user = Slides::find($slider_id);

        if (!$user) {
            return response()->json(
                [
                    'status' => false,
                    'message' => 'Slider not found',
                ],
                404
            );
        }

        // Handle image upload if provided
        if ($request->hasFile('image')) {
            $file = $request->file('image');
            $image_exts = ['tif', 'jpg', 'jpeg', 'gif', 'png'];
            $extension = $file->getClientOriginalExtension();

            if (!in_array($extension, $image_exts)) {
                return response()->json(
                    [
                        'status' => false,
                        'message' => 'Image Type Error',
                    ],
                    400
                );
            }

            // Delete the old image if it exists
            // $oldImagePath = public_path('/assets/images/slides/' . $user->image);
            // if ($user->image && file_exists($oldImagePath)) {
            //     unlink($oldImagePath);
            // }

            // Save the new image
            $fileName = 'image-' . uniqid() . '.' . $extension;
            $file->move(public_path('/assets/images/slides/'), $fileName);
            $image = $fileName;
        } else {
            $image = $user->image;
        }



        // Update user profile data
        $user->update([
            'description' => $request->input('description', $user->description),
            'status' => $request->input('status', $user->status),
            'image' => $image,
        ]);

        // Return updated user details with the full image URL or default image if none exists
        return response()->json(
            [
                'status' => true,
                'message' => 'Slider updated successfully ...!',
                'sliderdetails' => [
                    'id' => $user->id,
                    'description' => $user->description ?? '',
                    'status' => $user->status ?? '',
                    'image' => $user->image
                        ? url('assets/images/slides/' . $user->image)
                        : "",
                ],
            ],
            200
        );
    }

    public function serviceReviewList(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        $service_id = $request->service_id;
        $perPage = $request->input('per_page', 10); // Default: 10 items per page

        $reviewlist = ServiceReview::select(
            'service_review.id',
            'service_review.user_id',
            'service_review.service_id',
            'service_review.review_star',
            'service_review.review_message',
            'service_review.created_at',
            'services.service_name',
            'categories.category_name',
            'users.first_name',
            'users.last_name',
            'users.email',
            'users.image'
        )
            ->leftJoin('services', 'service_review.service_id', '=', 'services.id')
            ->leftJoin('categories', 'services.category_id', '=', 'categories.id')
            ->leftJoin('users', 'service_review.user_id', '=', 'users.id')
            ->where('service_review.service_id', $service_id)
            ->orderBy('service_review.created_at', 'desc')
            ->paginate($perPage);

        // Process the reviews
        foreach ($reviewlist as $item) {
            $item->time_ago = Carbon::parse($item->created_at)->diffForHumans();

            $service_images = ServiceImages::where('service_id', $item->service_id)
                ->pluck('service_images')
                ->toArray();

            $item->service_images = !empty($service_images)
                ? array_map(function ($image) {
                    return asset('assets/images/service_images/' . $image);
                }, $service_images)
                : [asset('assets/images/service_images/default_service.png')];

            $item->service_name = $item->service_name ?? '';
            $item->category_name = $item->category_name ?? '';
            $item->first_name = $item->first_name ?? '';
            $item->last_name = $item->last_name ?? '';
            $item->email = $item->email ?? '';
            $item->image = $item->image
                ? url('assets/images/users_images/' . $item->image)
                : "";
        }

        if ($reviewlist->isNotEmpty()) {
            return response()->json([
                'status' => true,
                'message' => 'Review list found',
                'reviewlist' => $reviewlist->items(), // Only review items
                'pagination' => [
                    'current_page' => $reviewlist->currentPage(),
                    'last_page' => $reviewlist->lastPage(),
                    'per_page' => $reviewlist->perPage(),
                    'total' => $reviewlist->total(),
                ],
            ]);
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Review list not found',
                'reviewlist' => [],
            ]);
        }
    }

    public function paymentconfiguration(Request $request)
    {
        try {
            // Always update ID = 1
            $siteSettings = SiteSetup::find(1);

            if (!$siteSettings) {
                return response()->json([
                    'status' => false,
                    'message' => 'Payment settings not found.',
                ], 404);
            }

            return response()->json([
                'status' => true,
                'message' => 'Payment settings successfully!',
                'is_payment' => $siteSettings->is_extended ? ($siteSettings->is_extended = 1 ? true : false) : false,
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while updating general settings: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function admin_all_login_status(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        if ($request->id) {
            $loginStatus = UserLoginStatus::find($request->id);
            $loginStatus->status = $request->status ?? "";
            $loginStatus->save();
        }
        $allStatuses = UserLoginStatus::get();

        $normalUsers = [];
        $adminUsers = [];

        foreach ($allStatuses as $status) {
            $item = [
                'id' => (string) $status->id,
                'text' => (string) $status->text,
                'status' => (string) $status->status,
            ];

            // if (in_array($status->id, [1, 2, 3])) {
            //     $normalUsers[] = $item;
            // } elseif (in_array($status->id, [4, 5])) {
            //     $adminUsers[] = $item;
            // }
            $normalUsers[] = $item;
        }

        // $groupedResponse = [
        //     'normal_data' => $normalUsers,
        //     // 'admin_data'  => $adminUsers,
        // ];

        // return $this->sendResponse($groupedResponse, "User Login Status List");

        $response = [
            "response_code" => "1",
            "message" => "Login status updated & fetched successfully",
            "status" => "success",
            "groupedResponse" => $normalUsers,
        ];

        return response()->json($response);
    }
    public function all_login_status(Request $request)
    {
        $google = UserLoginStatus::where('id', "2")->first();
        $apple = UserLoginStatus::where('id', "3")->first();
        $email = UserLoginStatus::where('id', "4")->first();
        $mobileotp = UserLoginStatus::where('id', "1")->first();

        $temp = [
            "response_code" => "1",
            "message" => "User Login Status Successfully",
            "status" => "success",
            "apple" => (string)$apple->status,
            "google" => (string)$google->status,
            "email" => (string)$email->status,
            "mobileotp" => (string)$mobileotp->status,
        ];

        return response()->json($temp);
    }
}
