import React, { useEffect, useState } from 'react';
import { MdOutlineChevronRight, MdOutlineCheck } from 'react-icons/md';
import { useDispatch } from 'react-redux';
import { UseGetCurrenyList } from '../../Appstore/Api/UseGetCurrenyList';
import { useAppSelector } from '../../Hooks/Hooks';
import { setSelectedCurrency } from '../../Appstore/Slice/currencySlice';

const CurrencyList = () => {
  const dispatch = useDispatch();
  const selectedCurrency = useAppSelector((state) => state.currency.selectedCurrency);
  const { data, isLoading } = UseGetCurrenyList();

  const [searchValue, setSearchValue] = useState('');
  const [showDropdown, setShowDropdown] = useState(false);
  const [rotate, setRotate] = useState(false);

  // Set default value from selected currency on mount or change
  useEffect(() => {
    if (selectedCurrency?.name) {
      setSearchValue(`${selectedCurrency.name}`);
    }
  }, [selectedCurrency]);

  const handleToggle = () => {
    setShowDropdown(!showDropdown);
    setRotate(!rotate);
  };

  const handleSelectCurrency = (currency: { name: string; currency: string }) => {
    dispatch(setSelectedCurrency(currency));
    setSearchValue(currency.name);
    setShowDropdown(false);
    setRotate(false);
  };

  const filteredCurrencies =
    data?.currencies?.filter((currency) =>
      currency.name.toLowerCase().includes(searchValue.toLowerCase())
    ) || [];

  return (
    <div className="w-full flex flex-col relative h-full">
      <label htmlFor="currency" className="font-poppins text-sm font-normal text-textcolor">
        Currency
        <span className="text-[#F21818] pl-[1px]">*</span>
      </label>

      <div className="relative mt-2 flex items-center cursor-pointer">
        <input
          type="text"
          id="currency"
          name="currency"
          autoComplete="off"
          spellCheck="false"
          className="w-full rounded-lg border border-bordercolor text-textcolor bg-primary px-4 py-2.5 my-1 placeholder:font-gilroy_regular placeholder:text-sm placeholder:text-textcolor placeholder:opacity-50 focus:outline-none focus:ring-1 focus:ring-header"
          placeholder="Select currency"
          value={searchValue}
          onFocus={() => setShowDropdown(true)}
          onBlur={() => setTimeout(() => setShowDropdown(false), 200)}
          onChange={(e) => setSearchValue(e.target.value)}
        />
        <div
          className={`absolute right-2 text-xl cursor-pointer transition-transform ${rotate ? 'rotate-90' : '-rotate-90'}`}
          onClick={handleToggle}
        >
          <MdOutlineChevronRight />
        </div>
      </div>

      {showDropdown && (
        <ul className="absolute top-[5.9rem] left-0 w-full bg-primary text-textcolor rounded-lg shadow-lg z-10 border border-bordercolor mt-1 max-h-60 overflow-auto">
          {filteredCurrencies.length > 0 ? (
            filteredCurrencies.map((currency) => (
              <li
                key={currency.id}
                className="px-4 py-2 cursor-pointer font-poppins hover:bg-gray-200 flex justify-between items-center"
                onMouseDown={() => handleSelectCurrency(currency)}
              >
                {currency.name} ({currency.currency})
                {selectedCurrency?.name === currency.name && (
                  <MdOutlineCheck className="text-[#6565657a] text-lg" />
                )}
              </li>
            ))
          ) : (
            <li className="px-4 py-2 font-poppins text-center text-gray-500">No currency found</li>
          )}
        </ul>
      )}
    </div>
  );
};

export default CurrencyList;
