import React, { useState, Fragment, useMemo, useEffect } from "react";
import {
  LineChart, Line, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer, Legend,
} from "recharts";
import Calendar from "../../assets/Images/bar_calendar.png";
import { Listbox, Transition } from "@headlessui/react";
import { CheckIcon, ChevronUpDownIcon } from "@heroicons/react/20/solid";
import { UsegetDashboardData } from "../../Appstore/Api/UsegetDashboardData";
import useApiPost from "../../Hooks/PostData";
import AvatarWithSpinner from "../../Componets/LoadingComponest";

// Format data for chart
const transformChartData = (days: string[], users: string[], vendors: string[]) => {
  return days.map((day, i) => ({
    name: day,
    userCount: Number(users[i]),
    vendorCount: Number(vendors[i]),
  }));
};





// Custom tooltip 
const CustomTooltip = ({ active, payload, label }: any) => {
  if (active && payload && payload.length > 1) {
    return (
      <div className="bg-white p-3 rounded shadow border text-sm font-poppins">
        <p className="text-gray-700 font-semibold">{label}</p>
        <p className="text-blue-500">Users: {payload[0]?.value}</p>
        <p className="text-green-500">Vendors: {payload[1]?.value}</p>
      </div>
    );
  }
  return null;
};

// Static month list
const monthOptions = [
  { name: "January", value: "01" }, { name: "February", value: "02" }, { name: "March", value: "03" },
  { name: "April", value: "04" }, { name: "May", value: "05" }, { name: "June", value: "06" },
  { name: "July", value: "07" }, { name: "August", value: "08" }, { name: "September", value: "09" },
  { name: "October", value: "10" }, { name: "November", value: "11" }, { name: "December", value: "12" },
];

function UserGetByYear() {




  const { postData, data, loading } = useApiPost()

  const chartData = data?.data?.chartData;




  useEffect(() => {


    const fetchDashboardData = async () => {
      try {
        const formData = new FormData();
        await postData("/dashboard-list", formData);
      } catch (error) {
        console.error("Failed to fetch dashboard data:", error);
      }
    };

    fetchDashboardData();
  }, []);


  // Years from 2025 to current year
  const currentYear = new Date().getFullYear();
  const yearOptions = Array.from({ length: currentYear - 2025 + 1 }, (_, i) => (2025 + i).toString());

  // Selected year & month state
  const [selectedYear, setSelectedYear] = useState<string>(chartData?.year || currentYear.toString());
  const [selectedMonth, setSelectedMonth] = useState<{ name: string; value: string }>(
    monthOptions.find((m) => m.name === chartData?.monthName) || monthOptions[new Date().getMonth()]
  );








  console.log("selectedMonthselectedMonthselectedMonthselectedMonth", selectedMonth.value)



  useEffect(() => {
    if (!selectedMonth || !selectedYear) return;

    const fetchDashboardData = async () => {
      try {
        const formData = new FormData();
        formData.append("month", selectedMonth.value);
        formData.append("year", selectedYear);

        await postData("/dashboard-list", formData);
      } catch (error) {
        console.error("Failed to fetch dashboard data:", error);
      }
    };

    fetchDashboardData();
  }, [selectedMonth, selectedYear]);




  const transformedData = useMemo(() => {
    if (!chartData) return [];
    return transformChartData(chartData.days, chartData.userCounts, chartData.vendorCounts);
  }, [chartData]);



  if (loading) return <div className="h-[400px]   w-full flex items-center justify-center">
    <AvatarWithSpinner />
  </div>;

  return (
    <div className="border border-bordercolor rounded-lg p-4 w-full">
      <div className="overflow-x-auto 2xl:overflow-x-visible w-full">
        <div className="flex justify-between gap-4 md:flex-row  flex-col items-center mb-2">
          <h2 className="text-textcolor  font-poppins text-base font-semibold">
            User & Vendor Growth (Line Chart)
          </h2>

          {/* Year + Month Dropdowns */}
          <div className="flex justify-between w-full md:w-fit   md:gap-4">
            {/* Year Dropdown */}
            <Dropdown
              label={selectedYear}
              options={yearOptions.map((y) => ({ label: y, value: y }))}
              selected={selectedYear}
              onChange={setSelectedYear}
            />

            {/* Month Dropdown */}
            <Dropdown
              label={selectedMonth.name}
              options={monthOptions.map((m) => ({ label: m.name, value: m.value }))}
              selected={selectedMonth.value}
              onChange={(val) => {
                const selected = monthOptions.find((m) => m.value === val);
                if (selected) setSelectedMonth(selected);
              }}
            />
          </div>
        </div>

        <p className="text-[#737373] font-poppins text-sm pb-12">
          {selectedMonth.name} {selectedYear} overview of Users and Vendors
        </p>

        <ResponsiveContainer width="100%" height={350}>
          <LineChart data={transformedData} margin={{ top: 5, right: 20, left: 10, bottom: 5 }}>
            <CartesianGrid strokeDasharray="3 3" vertical={false} />
            <XAxis dataKey="name" />
            <YAxis />
            <Tooltip content={<CustomTooltip />} />
            <Legend />
            <Line type="monotone" dataKey="userCount" stroke="#59A7FF" strokeWidth={2} name="Users" dot />
            <Line type="monotone" dataKey="vendorCount" stroke="#00C49F" strokeWidth={2} name="Vendors" dot />
          </LineChart>
        </ResponsiveContainer>
      </div>
    </div>
  );
}

export default UserGetByYear;


// reuse

const Dropdown = ({
  label,
  options,
  selected,
  onChange,
}: {
  label: string;
  options: { label: string; value: string }[];
  selected: string;
  onChange: (value: string) => void;
}) => {
  return (
    <Listbox value={selected} onChange={onChange}>
      <div className="relative w-[10rem]">
        <Listbox.Button className="relative w-full cursor-pointer rounded-md border border-bordercolor bg-primary py-2 pl-10 pr-10 text-left text-sm text-textcolor shadow-sm hover:border-gray-400 focus:outline-none">
          <img src={Calendar} className="absolute left-2 top-2.5 w-4 h-4" alt="Calendar" />
          <span>{label}</span>
          <span className="absolute inset-y-0 right-2 flex items-center pointer-events-none">
            <ChevronUpDownIcon className="h-5 w-5 text-gray-400" />
          </span>
        </Listbox.Button>

        <Transition
          as={Fragment}
          leave="transition ease-in duration-100"
          leaveFrom="opacity-100"
          leaveTo="opacity-0"
        >
          <Listbox.Options className="absolute rounded-lg z-10 mt-1 max-h-60 w-full overflow-auto  bg-primary py-1 text-sm shadow-lg ring-1 ring-black/5 focus:outline-none">
            {options.map((opt) => (
              <Listbox.Option
                key={opt.value}
                className={({ active }) =>
                  `cursor-pointer select-none py-2 relative pl-10 pr-4 ${active ? "bg-blue-100 text-[#239C57]" : "text-textcolor"}`
                }
                value={opt.value}
              >
                <>
                  <span className={`block truncate ${opt.value === selected ? "font-medium" : "font-normal"}`}>
                    {opt.label} 
                  </span>
                  {opt.value === selected && (
                    <span className="absolute inset-y-0 left-2 flex items-center text-[#239C57]">
                      <CheckIcon className="h-5 w-5" />
                    </span>
                  )}
                </>
              </Listbox.Option>
            ))}
          </Listbox.Options>
        </Transition>
      </div>
    </Listbox>
  );
};
