"use client";

import { useDispatch } from "react-redux";
import { useEffect, useState } from "react";
import { MdOutlineCheck, MdOutlineChevronRight } from "react-icons/md";
import { UseGetStoreList } from "../../Appstore/Api/UseGetStoreList";
import { useAppSelector } from "../../Hooks/Hooks";
import {
  toggleStoreSelection,
  removeStore,
} from "../../Appstore/Slice/storennameslicemultiple";

interface CategoryDropdownProps {
  required?: boolean;
}

const StoreDropdwon: React.FC<CategoryDropdownProps> = () => {
  const dispatch = useDispatch();
  const { data, isLoading } = UseGetStoreList();

  const selectedStores = useAppSelector(
    (state) => state.storennameslicemultiple.selectedStores
  );

  const categories =
    data?.ServiceList.map((item) => ({
      id: item.id,
      service_name: item.service_name,
    })) || [];

  const [searchValue, setSearchValue] = useState<string>("");
  const [showDropdown, setShowDropdown] = useState<boolean>(false);
  const [rotate, setRotate] = useState<boolean>(false);

  const handleSelectCategory = (categoryName: string, categoryId: number) => {
    dispatch(toggleStoreSelection({ id: categoryId, service_name: categoryName }));
    setShowDropdown(false);
    setRotate(false);
    setSearchValue(""); // Optional: clear input after selection
  };

  const handleFocus = () => {
    setShowDropdown(true);
    setRotate(true);
  };

  const handleBlur = () => {
    setTimeout(() => {
      setShowDropdown(false);
      setRotate(false);
    }, 100);
  };

  const handleToggle = () => {
    setShowDropdown((prev) => !prev);
    setRotate((prev) => !prev);
  };

  const isSelected = (id: number) =>
    selectedStores.some((cat) => cat.id === id);

  const filteredCategories = categories.filter((cat) =>
    cat.service_name.toLowerCase().includes(searchValue.toLowerCase())
  );

  return (
    <div className="w-full flex flex-col relative h-full">
      <label
        htmlFor="category"
        className="font-poppins text-sm font-medium text-textcolor"
      >
        Store
        <span className="text-[#F21818] pl-[1px]">*</span>
      </label>

      <div className="relative mt-1 flex items-center cursor-pointer">
        <input
          type="text"
          id="category"
          name="category"
          autoComplete="off"
          spellCheck="false"
          className="w-full rounded-lg border border-bordercolor cursor-pointer focus:border-toggalbtcolorborder text-textcolor bg-primary px-4 py-2.5 my-1 placeholder:text-placeholdercolor placeholder:text-sm focus:outline-none"
          placeholder="Search or select store"
          value={searchValue}
          onFocus={handleFocus}
          onBlur={handleBlur}
          onChange={(e) => setSearchValue(e.target.value)}
        />
        <div
          className={`absolute right-2 text-xl cursor-pointer   transition-transform ${rotate ? "rotate-90" : "-rotate-90"}`}
          onClick={handleToggle}
        >
          <MdOutlineChevronRight className="text-xl" />
        </div>
      </div>

      {selectedStores.length > 0 && (
        <div className="flex flex-wrap mt-2 gap-2">
          {selectedStores.map((store) => (
            <span
              key={store.id}
              className="flex items-center bg-[#1c5bb9c4] text-white px-3 py-1 rounded-full text-sm font-poppins"
            >
              {store.service_name}
              <button
                className="ml-2 text-red-500 cursor-pointer hover:text-red-700"
                onClick={() => dispatch(removeStore(store.id))}
              >
                ×
              </button>
            </span>
          ))}
        </div>
      )}

      {showDropdown && (
        <ul className="absolute top-[5rem] left-0 w-full bg-primary text-textcolor rounded-lg shadow-lg max-h-[200px] overflow-y-auto z-10 border border-bordercolor mt-1">
          {isLoading ? (
            <li className="px-4 py-2 font-poppins text-center text-gray-500">
              Loading stores...
            </li>
          ) : filteredCategories.length > 0 ? (
            filteredCategories.map((cat) => (
              <li
                key={cat.id}
                className="px-4 py-2 cursor-pointer font-poppins hover:bg-gray-200 text-textcolor flex justify-between items-center"
                onMouseDown={() => handleSelectCategory(cat.service_name, cat.id)}
              >
                {cat.service_name}
                {isSelected(cat.id) && (
                  <MdOutlineCheck className="text-[#6565657a] text-lg" />
                )}
              </li>
            ))
          ) : (
            <li className="px-4 py-2 font-poppins text-center text-gray-500">
              No stores found
            </li>
          )}
        </ul>
      )}
    </div>
  );
};

export default StoreDropdwon;
