import React, { useState } from 'react';
import { MdOutlineChevronRight, MdOutlineCheck } from 'react-icons/md';
import { useDispatch } from 'react-redux';
import { setDuration } from '../../../Appstore/Slice/AddVendorSlice';

const DurationDropdown = () => {
    const dispatch = useDispatch();

    const [searchValue, setSearchValue] = useState('');
    const [showDropdown, setShowDropdown] = useState(false);
    const [selectedDuration, setSelectedDuration] = useState('');
    const [rotate, setRotate] = useState(false);

    const durationOptions = ['1 Month', '3 Months', '6 Months', '1 Year'];

    const handleToggle = () => {
        setShowDropdown(!showDropdown);
        setRotate(!rotate);
    };

    const handleSelectDuration = (duration: string) => {
        setSelectedDuration(duration);
        setSearchValue(duration);
        setShowDropdown(false);
        setRotate(false);

        // 👇 Dispatch the exact string value
        dispatch(setDuration(duration));
    };

    const handleClear = () => {
        setSelectedDuration('');
        setSearchValue('');
        dispatch(setDuration(null)); // optional reset
    };

    return (
        <div className="w-full flex  gap-1 flex-col relative h-full">
            <label htmlFor="duration" className="font-poppins text-sm font-normal text-textcolor">
                Duration
                <span className="text-[#F21818] pl-[1px]">*</span>
            </label>

            <div className="relative  flex items-center cursor-pointer">
                <input
                    type="text"
                    id="duration"
                    name="duration"
                    autoComplete="off"
                    spellCheck="false"
                    className="w-full rounded-lg border border-bordercolor text-textcolor bg-primary px-4 py-2.5 my-1 placeholder:font-gilroy_regular placeholder:text-sm placeholder:text-textcolor placeholder:opacity-50 focus:outline-none focus:ring-1 focus:ring-header"
                    placeholder="Select duration"
                    value={searchValue}
                    onFocus={() => setShowDropdown(true)}
                    onBlur={() => setTimeout(() => setShowDropdown(false), 200)}
                    onChange={(e) => setSearchValue(e.target.value)}
                />
                <div
                    className={`absolute right-2 text-xl cursor-pointer transition-transform ${rotate ? 'rotate-90' : '-rotate-90'}`}
                    onClick={handleToggle}
                >
                    <MdOutlineChevronRight />
                </div>
            </div>

            {selectedDuration && (
                <div className="flex flex-wrap mt-2 gap-2">
                    <span className="flex items-center bg-[#1c5bb9c4] text-white px-3 py-1 rounded-full text-sm font-poppins">
                        {selectedDuration}
                        <button className="ml-2 text-red-500 hover:text-red-700" onClick={handleClear}>×</button>
                    </span>
                </div>
            )}

            {showDropdown && (
                <ul className="absolute top-[5.9rem] left-0 w-full bg-primary text-textcolor rounded-lg shadow-lg z-10 border border-bordercolor mt-1">
                    {durationOptions
                        .filter((d) => d.toLowerCase().includes(searchValue.toLowerCase()))
                        .map((duration) => (
                            <li
                                key={duration}
                                className="px-4 py-2 cursor-pointer font-poppins hover:bg-gray-200 flex justify-between items-center"
                                onMouseDown={() => handleSelectDuration(duration)}
                            >
                                {duration}
                                {selectedDuration === duration && (
                                    <MdOutlineCheck className="text-[#6565657a] text-lg" />
                                )}
                            </li>
                        ))}
                    {durationOptions.filter((d) =>
                        d.toLowerCase().includes(searchValue.toLowerCase())
                    ).length === 0 && (
                        <li className="px-4 py-2 font-poppins text-center text-gray-500">No duration found</li>
                    )}
                </ul>
            )}
        </div>
    );
};

export default DurationDropdown;
